<?php
/**
 * Frontend display: Shortcode + Template Tags for AEO generated Overview & FAQs.
 */

if ( ! defined( 'WPINC' ) ) { die; }

// Prevent duplicate schema injection tracking.
if ( ! defined( 'AEO_ALLOW_SHORTCODE_SCHEMA' ) ) {
	// Site owners can define this to true in wp-config.php if they want shortcode-driven schema.
	define( 'AEO_ALLOW_SHORTCODE_SCHEMA', false );
}

/**
 * Register shortcode on init.
 */
function aeo_register_shortcodes() {
	add_shortcode( 'aeo_content', 'aeo_shortcode_render_content' );
}
add_action( 'init', 'aeo_register_shortcodes' );

/**
 * Register auto-insert filter late so builders can finish content handling.
 */
function aeo_register_auto_insert_filter() {
	$priority = (int) apply_filters( 'aeo_auto_insert_priority', PHP_INT_MAX - 1 );
	add_filter( 'the_content', 'aeo_auto_insert_content', $priority );
}
add_action( 'init', 'aeo_register_auto_insert_filter' );

/**
 * Register render_block fallback for block themes/builders.
 */
function aeo_register_block_auto_insert_filter() {
	$priority = (int) apply_filters( 'aeo_auto_insert_block_priority', PHP_INT_MAX - 1 );
	add_filter( 'render_block', 'aeo_render_block_auto_insert', $priority, 2 );
}
add_action( 'init', 'aeo_register_block_auto_insert_filter' );

/**
 * Register Elementor fallback filter for builder-rendered content.
 */
function aeo_register_elementor_auto_insert_filter() {
	$enable = (bool) apply_filters( 'aeo_enable_elementor_frontend_filter', false );
	if ( ! $enable ) {
		return;
	}
	$priority = (int) apply_filters( 'aeo_auto_insert_elementor_priority', PHP_INT_MAX - 1 );
	add_filter( 'elementor/frontend/the_content', 'aeo_elementor_auto_insert_content', $priority );
}
add_action( 'init', 'aeo_register_elementor_auto_insert_filter' );

/**
 * Register Elementor widget-level filter for Post Content widget.
 */
function aeo_register_elementor_widget_auto_insert_filter() {
	$priority = (int) apply_filters( 'aeo_auto_insert_elementor_widget_priority', PHP_INT_MAX - 1 );
	add_filter( 'elementor/widget/render_content', 'aeo_elementor_widget_auto_insert', $priority, 2 );
}
add_action( 'init', 'aeo_register_elementor_widget_auto_insert_filter' );

/**
 * Determine which post types are eligible for auto insert.
 */
function aeo_get_auto_insert_post_types() {
	$types = apply_filters( 'aeo_auto_insert_post_types', array( 'post' ) );
	if ( ! is_array( $types ) ) {
		$types = array( 'post' );
	}
	return array_values( array_filter( array_map( 'sanitize_key', $types ) ) );
}

/**
 * Resolve the target post ID for auto insert on the current request.
 */
function aeo_get_auto_insert_context_post_id( $post = null ) {
	if ( is_admin() || ! is_singular() ) {
		return 0;
	}
	$queried_id = get_queried_object_id();
	if ( $queried_id ) {
		return (int) $queried_id;
	}
	if ( $post instanceof WP_Post ) {
		return (int) $post->ID;
	}
	return 0;
}

/**
 * Check whether the post type is eligible for auto insert.
 */
function aeo_is_auto_insert_supported_post_type( $post_id ) {
	$post_type = get_post_type( $post_id );
	if ( ! $post_type ) {
		return false;
	}
	$types = aeo_get_auto_insert_post_types();
	if ( empty( $types ) ) {
		return false;
	}
	return in_array( $post_type, $types, true );
}

/**
 * Track auto-insert usage per request to prevent duplicates.
 */
function aeo_auto_insert_is_inserted( $post_id, $source = '' ) {
	$post_id = (int) $post_id;
	if ( $post_id < 1 ) {
		return false;
	}
	if ( empty( $GLOBALS['aeo_auto_inserted'] ) || ! is_array( $GLOBALS['aeo_auto_inserted'] ) ) {
		$GLOBALS['aeo_auto_inserted'] = array();
	}
	$source = $source !== '' ? (string) $source : aeo_get_auto_insert_source();
	if ( empty( $GLOBALS['aeo_auto_inserted'][ $post_id ] ) || ! is_array( $GLOBALS['aeo_auto_inserted'][ $post_id ] ) ) {
		return false;
	}
	return ! empty( $GLOBALS['aeo_auto_inserted'][ $post_id ][ $source ] );
}

function aeo_auto_insert_mark_inserted( $post_id, $source = '' ) {
	$post_id = (int) $post_id;
	if ( $post_id < 1 ) {
		return;
	}
	if ( empty( $GLOBALS['aeo_auto_inserted'] ) || ! is_array( $GLOBALS['aeo_auto_inserted'] ) ) {
		$GLOBALS['aeo_auto_inserted'] = array();
	}
	$source = $source !== '' ? (string) $source : aeo_get_auto_insert_source();
	if ( empty( $GLOBALS['aeo_auto_inserted'][ $post_id ] ) || ! is_array( $GLOBALS['aeo_auto_inserted'][ $post_id ] ) ) {
		$GLOBALS['aeo_auto_inserted'][ $post_id ] = array();
	}
	$GLOBALS['aeo_auto_inserted'][ $post_id ][ $source ] = true;
}

function aeo_get_auto_insert_source() {
	return isset( $GLOBALS['aeo_auto_insert_source'] ) ? (string) $GLOBALS['aeo_auto_insert_source'] : 'the_content';
}

function aeo_set_auto_insert_source( $source ) {
	$GLOBALS['aeo_auto_insert_source'] = (string) $source;
}

function aeo_clear_auto_insert_source() {
	unset( $GLOBALS['aeo_auto_insert_source'] );
}

function aeo_auto_insert_content_has_marker( $content ) {
	if ( ! is_string( $content ) || $content === '' ) {
		return false;
	}
	return ( false !== strpos( $content, 'aeo-auto-inserted' ) || false !== strpos( $content, 'aeo-content' ) );
}

function aeo_is_elementor_template_post( $post ) {
	if ( ! ( $post instanceof WP_Post ) ) {
		return false;
	}
	return ( 'elementor_library' === $post->post_type );
}

/**
 * Debug helpers (enable via AEO_DEBUG_FRONTEND in wp-config.php).
 */
function aeo_debug_frontend_request_flag() {
	$debug = (string) filter_input( INPUT_GET, 'aeo_debug', FILTER_UNSAFE_RAW );
	if ( '' === $debug ) {
		return false;
	}
	$nonce_raw = (string) filter_input( INPUT_GET, 'aeo_debug_nonce', FILTER_UNSAFE_RAW );
	$nonce = sanitize_text_field( $nonce_raw );
	if ( '' === $nonce ) {
		return false;
	}
	$nonce_action = (string) apply_filters( 'aeo_debug_frontend_nonce_action', 'aeo_debug_frontend' );
	return (bool) wp_verify_nonce( $nonce, $nonce_action );
}

function aeo_debug_frontend_enabled() {
	$enabled = defined( 'AEO_DEBUG_FRONTEND' ) && AEO_DEBUG_FRONTEND;
	if ( ! $enabled && aeo_debug_frontend_request_flag() ) {
		$enabled = is_user_logged_in() && current_user_can( 'manage_options' );
	}
	return (bool) apply_filters( 'aeo_debug_frontend_enabled', $enabled );
}

function aeo_debug_frontend_state_init( $post_id ) {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	$post_id = (int) $post_id;
	if ( $post_id < 1 ) {
		return;
	}
	if ( empty( $GLOBALS['aeo_debug_frontend_state'] ) || ! is_array( $GLOBALS['aeo_debug_frontend_state'] ) ) {
		$GLOBALS['aeo_debug_frontend_state'] = array();
	}
	if ( isset( $GLOBALS['aeo_debug_frontend_state'][ $post_id ] ) ) {
		return;
	}
	$GLOBALS['aeo_debug_frontend_state'][ $post_id ] = array(
		'post_id' => $post_id,
		'post_type' => (string) get_post_type( $post_id ),
		'called' => array(),
		'inserted' => false,
		'skip_reason' => '',
		'auto_summary' => (int) get_option( 'aeo_auto_insert_summary', 1 ),
		'auto_faqs' => (int) get_option( 'aeo_auto_insert_faqs', 1 ),
		'has_overview' => null,
		'faqs_count' => null,
	);
}

function aeo_debug_frontend_set_state( $post_id, $key, $value ) {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	$post_id = (int) $post_id;
	if ( $post_id < 1 ) {
		return;
	}
	aeo_debug_frontend_state_init( $post_id );
	$GLOBALS['aeo_debug_frontend_state'][ $post_id ][ $key ] = $value;
}

function aeo_debug_frontend_add_called( $post_id, $source ) {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	$post_id = (int) $post_id;
	if ( $post_id < 1 ) {
		return;
	}
	aeo_debug_frontend_state_init( $post_id );
	if ( empty( $GLOBALS['aeo_debug_frontend_state'][ $post_id ]['called'] ) ) {
		$GLOBALS['aeo_debug_frontend_state'][ $post_id ]['called'] = array();
	}
	if ( ! in_array( $source, $GLOBALS['aeo_debug_frontend_state'][ $post_id ]['called'], true ) ) {
		$GLOBALS['aeo_debug_frontend_state'][ $post_id ]['called'][] = $source;
	}
}

function aeo_debug_frontend_set_skip_reason( $post_id, $reason ) {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	$post_id = (int) $post_id;
	if ( $post_id < 1 ) {
		return;
	}
	aeo_debug_frontend_state_init( $post_id );
	if ( empty( $GLOBALS['aeo_debug_frontend_state'][ $post_id ]['skip_reason'] ) ) {
		$GLOBALS['aeo_debug_frontend_state'][ $post_id ]['skip_reason'] = (string) $reason;
	}
}

function aeo_debug_frontend_append_comment( $content, $note, $post_id = 0 ) {
	if ( ! aeo_debug_frontend_enabled() ) {
		return $content;
	}
	$key = $post_id > 0 ? (string) $post_id : 'none';
	if ( empty( $GLOBALS['aeo_debug_frontend_appended'] ) || ! is_array( $GLOBALS['aeo_debug_frontend_appended'] ) ) {
		$GLOBALS['aeo_debug_frontend_appended'] = array();
	}
	if ( ! empty( $GLOBALS['aeo_debug_frontend_appended'][ $key ] ) ) {
		return $content;
	}
	$GLOBALS['aeo_debug_frontend_appended'][ $key ] = true;
	$comment = '<!-- AEO DEBUG: ' . esc_html( $note ) . ' -->';
	return $content . "\n" . $comment;
}

function aeo_debug_frontend_filter_stats( $filter_name, $target_callback ) {
	global $wp_filter;
	$stats = array(
		'count' => 0,
		'max_priority' => 0,
		'has_target' => false,
	);
	if ( empty( $wp_filter[ $filter_name ] ) ) {
		return $stats;
	}
	$hook = $wp_filter[ $filter_name ];
	$callbacks = $hook instanceof WP_Hook ? $hook->callbacks : $hook;
	if ( empty( $callbacks ) || ! is_array( $callbacks ) ) {
		return $stats;
	}
	foreach ( $callbacks as $priority => $set ) {
		$stats['max_priority'] = max( $stats['max_priority'], (int) $priority );
		if ( ! is_array( $set ) ) {
			continue;
		}
		foreach ( $set as $cb ) {
			$stats['count']++;
			if ( empty( $cb['function'] ) ) {
				continue;
			}
			$fn = $cb['function'];
			if ( is_string( $fn ) && $fn === $target_callback ) {
				$stats['has_target'] = true;
			} elseif ( is_array( $fn ) && isset( $fn[1] ) && $fn[1] === $target_callback ) {
				$stats['has_target'] = true;
			}
		}
	}
	return $stats;
}

function aeo_debug_frontend_init() {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	$post_id = aeo_get_auto_insert_context_post_id();
	if ( $post_id ) {
		aeo_debug_frontend_state_init( $post_id );
		$GLOBALS['aeo_debug_frontend_context_post'] = (int) $post_id;
	}
}
add_action( 'wp', 'aeo_debug_frontend_init' );

function aeo_debug_frontend_footer_comment() {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	$post_id = isset( $GLOBALS['aeo_debug_frontend_context_post'] ) ? (int) $GLOBALS['aeo_debug_frontend_context_post'] : 0;
	if ( $post_id < 1 ) {
		echo "<!-- AEO DEBUG: no singular context post -->\n";
		return;
	}
	$state = isset( $GLOBALS['aeo_debug_frontend_state'][ $post_id ] ) ? $GLOBALS['aeo_debug_frontend_state'][ $post_id ] : array();
	$called = isset( $state['called'] ) && is_array( $state['called'] ) ? implode( ',', $state['called'] ) : '';
	$skip = isset( $state['skip_reason'] ) ? (string) $state['skip_reason'] : '';
	$inserted = ! empty( $state['inserted'] ) ? '1' : '0';
	$auto_summary = isset( $state['auto_summary'] ) ? (int) $state['auto_summary'] : 0;
	$auto_faqs = isset( $state['auto_faqs'] ) ? (int) $state['auto_faqs'] : 0;
	$has_overview = isset( $state['has_overview'] ) ? (string) $state['has_overview'] : 'null';
	$faqs_count = isset( $state['faqs_count'] ) ? (string) $state['faqs_count'] : 'null';
	$content_stats = aeo_debug_frontend_filter_stats( 'the_content', 'aeo_auto_insert_content' );
	$block_stats = aeo_debug_frontend_filter_stats( 'render_block', 'aeo_render_block_auto_insert' );
	$elementor_stats = aeo_debug_frontend_filter_stats( 'elementor/frontend/the_content', 'aeo_elementor_auto_insert_content' );
	$elementor_widget_stats = aeo_debug_frontend_filter_stats( 'elementor/widget/render_content', 'aeo_elementor_widget_auto_insert' );
	$should_stats = aeo_debug_frontend_filter_stats( 'aeo_auto_insert_should_render', '' );
	$parts = array(
		'context_post=' . $post_id,
		'type=' . ( isset( $state['post_type'] ) ? $state['post_type'] : '' ),
		'called=' . $called,
		'inserted=' . $inserted,
		'skip=' . $skip,
		'auto_summary=' . $auto_summary,
		'auto_faqs=' . $auto_faqs,
		'has_overview=' . $has_overview,
		'faqs_count=' . $faqs_count,
		'the_content_count=' . $content_stats['count'],
		'the_content_max=' . $content_stats['max_priority'],
		'the_content_has_aeo=' . ( $content_stats['has_target'] ? '1' : '0' ),
		'render_block_count=' . $block_stats['count'],
		'render_block_max=' . $block_stats['max_priority'],
		'render_block_has_aeo=' . ( $block_stats['has_target'] ? '1' : '0' ),
		'elementor_count=' . $elementor_stats['count'],
		'elementor_max=' . $elementor_stats['max_priority'],
		'elementor_has_aeo=' . ( $elementor_stats['has_target'] ? '1' : '0' ),
		'elementor_widget_count=' . $elementor_widget_stats['count'],
		'elementor_widget_max=' . $elementor_widget_stats['max_priority'],
		'elementor_widget_has_aeo=' . ( $elementor_widget_stats['has_target'] ? '1' : '0' ),
		'auto_insert_should_render_count=' . $should_stats['count'],
		'auto_insert_should_render_max=' . $should_stats['max_priority'],
	);
	echo '<!-- AEO DEBUG: ' . esc_html( implode( ' | ', $parts ) ) . " -->\n";

	$payload = array(
		'context_post' => $post_id,
		'post_type' => isset( $state['post_type'] ) ? $state['post_type'] : '',
		'called' => $called,
		'inserted' => $inserted,
		'skip' => $skip,
		'auto_summary' => $auto_summary,
		'auto_faqs' => $auto_faqs,
		'has_overview' => $has_overview,
		'faqs_count' => $faqs_count,
		'filters' => array(
			'the_content' => $content_stats,
			'render_block' => $block_stats,
			'elementor' => $elementor_stats,
			'elementor_widget' => $elementor_widget_stats,
			'auto_insert_should_render' => $should_stats,
		),
	);
	echo '<script type="application/json" id="aeo-debug-data">' . wp_json_encode( $payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . "</script>\n";
}
add_action( 'wp_footer', 'aeo_debug_frontend_footer_comment', 99 );

function aeo_debug_frontend_head_comment() {
	if ( ! aeo_debug_frontend_enabled() ) {
		return;
	}
	if ( is_admin() || is_feed() ) {
		return;
	}
	$post_id = aeo_get_auto_insert_context_post_id();
	$type = $post_id ? (string) get_post_type( $post_id ) : '';
	$is_singular = is_singular() ? '1' : '0';
	$parts = array(
		'head',
		'context_post=' . ( $post_id ? $post_id : '0' ),
		'type=' . $type,
		'is_singular=' . $is_singular,
	);
	$meta_content = 'ctx=' . ( $post_id ? $post_id : '0' ) . ';type=' . $type . ';singular=' . $is_singular;
	echo '<meta name="aeo-debug" content="' . esc_attr( $meta_content ) . "\" />\n";
	echo '<!-- AEO DEBUG: ' . esc_html( implode( ' | ', $parts ) ) . " -->\n";
}
add_action( 'wp_head', 'aeo_debug_frontend_head_comment', 1 );

function aeo_debug_frontend_header() {
	$debug_request = aeo_debug_frontend_request_flag();
	if ( ! aeo_debug_frontend_enabled() && ! $debug_request ) {
		return;
	}
	if ( is_admin() || is_feed() || ( defined( 'REST_REQUEST' ) && REST_REQUEST ) ) {
		return;
	}
	if ( $debug_request && ! defined( 'DONOTCACHEPAGE' ) ) {
		define( 'DONOTCACHEPAGE', true );
	}
	if ( $debug_request ) {
		nocache_headers();
	}
	$post_id = aeo_get_auto_insert_context_post_id();
	$type = $post_id ? (string) get_post_type( $post_id ) : '';
	$parts = array(
		'loaded=1',
		'ctx=' . ( $post_id ? $post_id : 0 ),
		'type=' . $type,
		'singular=' . ( is_singular() ? 1 : 0 ),
		'elementor=' . ( defined( 'ELEMENTOR_VERSION' ) ? '1' : '0' ),
	);
	header( 'X-AEO-Debug: ' . implode( ';', $parts ) );
}
add_action( 'send_headers', 'aeo_debug_frontend_header', 10 );

/**
 * Auto insert overview + FAQs at render time (single post) when option enabled.
 */
function aeo_build_auto_insert_sections( $post_id ) {
	$auto_summary = (int) get_option( 'aeo_auto_insert_summary', 1 ) === 1;
	$auto_faqs    = (int) get_option( 'aeo_auto_insert_faqs', 1 ) === 1;
	$meta = aeo_get_content_meta( $post_id );
	$overview = $meta['overview'];
	$faqs = $meta['faqs'];
	$has_overview = '' !== trim( (string) $overview );
	$faqs_count = is_array( $faqs ) ? count( $faqs ) : 0;
	$overview_wrapper = '';
	$faqs_wrapper = '';

	if ( $auto_summary && $has_overview ) {
		$args_overview = array(
			'heading_overview' => 2,
			'title_overview' => __( 'Summary', 'anslift' ),
			'overview_accordion' => true,
			'accordion' => true,
		);
		$section = aeo_build_overview_markup( $overview, $args_overview );
		$overview_wrapper = '<div class="aeo-content aeo-mode-overview aeo-auto-inserted">' . $section . '</div>';
	}

	if ( $auto_faqs && $faqs_count > 0 ) {
		$args_faqs = array(
			'heading_faqs' => 2,
			'title_faqs' => __( 'Frequently Asked Questions', 'anslift' ),
			'faq_accordion' => true,
			'accordion' => true,
			'limit' => 0,
			'list_style' => 'dl',
		);
		$section_f = aeo_build_faqs_markup( $faqs, $args_faqs );
		$faqs_wrapper = '<div class="aeo-content aeo-mode-faqs aeo-auto-inserted">' . $section_f . '</div>';
	}

	return array(
		'overview_wrapper' => $overview_wrapper,
		'faqs_wrapper' => $faqs_wrapper,
		'auto_summary' => $auto_summary,
		'auto_faqs' => $auto_faqs,
		'has_overview' => $has_overview,
		'faqs_count' => $faqs_count,
	);
}

function aeo_auto_insert_content( $content ) {
	global $post;
	$post_id = aeo_get_auto_insert_context_post_id( $post );
	if ( ! $post_id ) {
		return aeo_debug_frontend_append_comment( $content, 'skip: no context post', 0 );
	}
	$source = aeo_get_auto_insert_source();
	if ( 'elementor' === $source && $post instanceof WP_Post && aeo_is_elementor_template_post( $post ) && (int) $post->ID !== (int) $post_id ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: elementor template (non-post)' );
		return aeo_debug_frontend_append_comment( $content, 'skip: elementor template (non-post)', $post_id );
	}
	if ( $post instanceof WP_Post && (int) $post->ID !== (int) $post_id ) {
		if ( 'the_content' === $source && defined( 'ELEMENTOR_VERSION' ) && aeo_is_elementor_template_post( $post ) ) {
			aeo_debug_frontend_set_skip_reason( $post_id, 'skip: elementor template content' );
			return aeo_debug_frontend_append_comment( $content, 'skip: elementor template content', $post_id );
		}
		if ( in_the_loop() ) {
			aeo_debug_frontend_set_skip_reason( $post_id, 'skip: global post mismatch in loop' );
			return aeo_debug_frontend_append_comment( $content, 'skip: global post mismatch in loop', $post_id );
		}
	}
	if ( ! aeo_is_auto_insert_supported_post_type( $post_id ) ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: unsupported post type' );
		return aeo_debug_frontend_append_comment( $content, 'skip: unsupported post type', $post_id );
	}
	if ( post_password_required( $post_id ) ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: password required' );
		return aeo_debug_frontend_append_comment( $content, 'skip: password required', $post_id );
	}

	aeo_debug_frontend_state_init( $post_id );
	aeo_debug_frontend_add_called( $post_id, aeo_get_auto_insert_source() );

	if ( aeo_auto_insert_content_has_marker( $content ) ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: marker already present' );
		return aeo_debug_frontend_append_comment( $content, 'skip: marker already present', $post_id );
	}
	if ( aeo_auto_insert_is_inserted( $post_id, $source ) ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: already inserted for source' );
		return aeo_debug_frontend_append_comment( $content, 'skip: already inserted for source', $post_id );
	}

	$should_insert = (bool) apply_filters( 'aeo_auto_insert_should_render', true, $post_id, $content );
	if ( ! $should_insert ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: filtered' );
		return aeo_debug_frontend_append_comment( $content, 'skip: filtered', $post_id );
	}

	$sections = aeo_build_auto_insert_sections( $post_id );
	aeo_debug_frontend_set_state( $post_id, 'has_overview', $sections['has_overview'] ? '1' : '0' );
	aeo_debug_frontend_set_state( $post_id, 'faqs_count', $sections['faqs_count'] );

	if ( ! $sections['auto_summary'] && ! $sections['auto_faqs'] ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: auto insert disabled' );
		return aeo_debug_frontend_append_comment( $content, 'skip: auto insert disabled', $post_id );
	}

	if ( ! $sections['overview_wrapper'] && ! $sections['faqs_wrapper'] ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: no overview/faqs content' );
		return aeo_debug_frontend_append_comment( $content, 'skip: no overview/faqs content', $post_id );
	}
	// Enqueue assets (JS needed for accordion toggling) mirroring shortcode behavior
	aeo_enqueue_frontend_assets( true );
	aeo_auto_insert_mark_inserted( $post_id, $source );
	aeo_debug_frontend_set_state( $post_id, 'inserted', true );
	return $sections['overview_wrapper'] . $content . $sections['faqs_wrapper'];
}

/**
 * Block theme fallback: inject around core/post-content block.
 */
function aeo_render_block_auto_insert( $block_content, $block ) {
	if ( empty( $block['blockName'] ) || 'core/post-content' !== $block['blockName'] ) {
		return $block_content;
	}
	$post_id = aeo_get_auto_insert_context_post_id();
	if ( ! $post_id ) {
		return $block_content;
	}
	if ( ! aeo_is_auto_insert_supported_post_type( $post_id ) ) {
		return $block_content;
	}
	if ( post_password_required( $post_id ) ) {
		return $block_content;
	}

	$source = 'render_block';
	aeo_set_auto_insert_source( $source );
	aeo_debug_frontend_state_init( $post_id );
	aeo_debug_frontend_add_called( $post_id, 'render_block' );

	if ( aeo_auto_insert_content_has_marker( $block_content ) ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: marker already present' );
		aeo_clear_auto_insert_source();
		return $block_content;
	}
	if ( aeo_auto_insert_is_inserted( $post_id, $source ) ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: already inserted for source' );
		aeo_clear_auto_insert_source();
		return $block_content;
	}

	$should_insert = (bool) apply_filters( 'aeo_auto_insert_should_render', true, $post_id, $block_content );
	if ( ! $should_insert ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: filtered' );
		aeo_clear_auto_insert_source();
		return $block_content;
	}

	$sections = aeo_build_auto_insert_sections( $post_id );
	aeo_debug_frontend_set_state( $post_id, 'has_overview', $sections['has_overview'] ? '1' : '0' );
	aeo_debug_frontend_set_state( $post_id, 'faqs_count', $sections['faqs_count'] );

	if ( ! $sections['auto_summary'] && ! $sections['auto_faqs'] ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: auto insert disabled' );
		aeo_clear_auto_insert_source();
		return $block_content;
	}

	if ( ! $sections['overview_wrapper'] && ! $sections['faqs_wrapper'] ) {
		aeo_debug_frontend_set_skip_reason( $post_id, 'skip: no overview/faqs content' );
		aeo_clear_auto_insert_source();
		return $block_content;
	}

	aeo_enqueue_frontend_assets( true );
	aeo_auto_insert_mark_inserted( $post_id, $source );
	aeo_debug_frontend_set_state( $post_id, 'inserted', true );

	$final = $sections['overview_wrapper'] . $block_content . $sections['faqs_wrapper'];
	aeo_clear_auto_insert_source();
	return $final;
}

function aeo_elementor_auto_insert_content( $content ) {
	aeo_set_auto_insert_source( 'elementor' );
	$result = aeo_auto_insert_content( $content );
	aeo_clear_auto_insert_source();
	return $result;
}

function aeo_elementor_widget_auto_insert( $content, $widget ) {
	if ( ! is_object( $widget ) || ! method_exists( $widget, 'get_name' ) ) {
		return $content;
	}
	$widget_name = (string) $widget->get_name();
	$allowed = apply_filters( 'aeo_elementor_content_widget_names', array( 'theme-post-content', 'post-content' ) );
	if ( ! is_array( $allowed ) || empty( $allowed ) ) {
		return $content;
	}
	if ( ! in_array( $widget_name, $allowed, true ) ) {
		return $content;
	}
	aeo_set_auto_insert_source( 'elementor_widget' );
	$result = aeo_auto_insert_content( $content );
	aeo_clear_auto_insert_source();
	return $result;
}

/**
 * Public helper to enqueue frontend assets (can be called manually by themes before outputting template tag).
 */
function aeo_enqueue_frontend_assets( $needs_js = false ) {
	// Allow disabling via filter.
	if ( false === apply_filters( 'aeo_enqueue_frontend_assets', true ) ) { return; }
	wp_enqueue_style( 'aeo-frontend-style', AEO_PLUGIN_URL . 'public/assets/frontend.css', array(), '1.0.0' );
	if ( $needs_js ) {
		wp_enqueue_script( 'aeo-frontend-script', AEO_PLUGIN_URL . 'public/assets/frontend.js', array(), '1.0.0', true );
	}
}

/**
 * Normalize shortcode / template args.
 */
function aeo_normalize_args( $atts ) {
	$defaults = array(
		'show' => 'both',
		'post_id' => 0,
	'title_overview' => __( 'Summary', 'anslift' ),
	'title_faqs' => __( 'Frequently Asked Questions', 'anslift' ),
		'heading_overview' => 2,
		'heading_faqs' => 2,
		'limit' => 0,
		'list_style' => 'dl', // dl|ol|ul|div
		'accordion' => 'false',
		// New granular control: Overview accordion independent of FAQ accordion. Defaults to true (closed by default)
		'overview_accordion' => 'true',
		'faq_accordion' => '', // fallback to legacy "accordion" flag when empty
		'class' => '',
		'show_schema' => 'false',
		'empty_behavior' => 'hide', // hide|message|placeholder
		'cache' => 0,
	);
	$atts = shortcode_atts( $defaults, $atts, 'aeo_content' );

	$atts['show'] = in_array( $atts['show'], array( 'overview', 'faqs', 'both' ), true ) ? $atts['show'] : 'both';
	$atts['post_id'] = intval( $atts['post_id'] );
	$atts['heading_overview'] = max( 2, min( 6, intval( $atts['heading_overview'] ) ) );
	$atts['heading_faqs'] = max( 2, min( 6, intval( $atts['heading_faqs'] ) ) );
	$atts['limit'] = intval( $atts['limit'] );
	$atts['list_style'] = in_array( $atts['list_style'], array( 'dl', 'ol', 'ul', 'div' ), true ) ? $atts['list_style'] : 'dl';
	$atts['accordion'] = filter_var( $atts['accordion'], FILTER_VALIDATE_BOOLEAN ); // legacy combined flag
	$atts['overview_accordion'] = isset( $atts['overview_accordion'] ) ? filter_var( $atts['overview_accordion'], FILTER_VALIDATE_BOOLEAN ) : true;
	// If faq_accordion explicitly set use it; else inherit old accordion flag
	$atts['faq_accordion'] = ( $atts['faq_accordion'] !== '' ) ? filter_var( $atts['faq_accordion'], FILTER_VALIDATE_BOOLEAN ) : $atts['accordion'];
	$atts['show_schema'] = filter_var( $atts['show_schema'], FILTER_VALIDATE_BOOLEAN );
	$atts['empty_behavior'] = in_array( $atts['empty_behavior'], array( 'hide', 'message', 'placeholder' ), true ) ? $atts['empty_behavior'] : 'hide';
	$atts['cache'] = max( 0, intval( $atts['cache'] ) );
	$atts['class'] = sanitize_html_class( $atts['class'], '' );

	return apply_filters( 'aeo_shortcode_pre_render_args', $atts );
}

/**
 * Retrieve overview + FAQs meta.
 */
function aeo_get_content_meta( $post_id ) {
	$overview = get_post_meta( $post_id, '_aeo_ai_overview', true );
	$faqs = get_post_meta( $post_id, '_aeo_faqs', true );
	if ( ! is_array( $faqs ) ) { $faqs = array(); }

	// Sanitize structure (keep raw text — output will escape later)
	$clean_faqs = array();
	foreach ( $faqs as $faq ) {
		if ( empty( $faq['question'] ) || empty( $faq['answer'] ) ) { continue; }
		$clean_faqs[] = array(
			'question' => (string) $faq['question'],
			'answer' => (string) $faq['answer'],
		);
	}
	return apply_filters( 'aeo_shortcode_raw_data', array(
		'overview' => $overview,
		'faqs' => $clean_faqs,
	), $post_id );
}

/**
 * Build overview markup.
 */
function aeo_build_overview_markup( $overview, $args ) {
	if ( '' === trim( (string) $overview ) ) { return ''; }
	$h = intval( $args['heading_overview'] );
	$title = trim( (string) $args['title_overview'] );
	// Overview now has its own accordion control (defaults true). Falls back to legacy combined 'accordion' flag.
	$accordion = isset( $args['overview_accordion'] ) ? (bool) $args['overview_accordion'] : ! empty( $args['accordion'] );
	$uid = uniqid( 'aeo_overview_', true );
	if ( $accordion ) {
		$out = '<section class="aeo-overview aeo-accordion-item" data-accordion="true">';
		// Use button instead of heading for collapsible control. Provide fallback heading semantics via aria-label.
		if ( $title !== '' ) {
			$out .= '<button type="button" class="aeo-overview-button aeo-accordion-trigger" role="heading" aria-level="' . $h . '" aria-expanded="false" aria-controls="' . esc_attr( $uid ) . '" aria-label="' . esc_attr( $title ) . '">';
			$out .= esc_html( $title );
			$out .= '</button>';
		}
		$out .= '<div id="' . esc_attr( $uid ) . '" class="aeo-overview-text aeo-accordion-panel" hidden>' . esc_html( $overview ) . '</div>';
		$out .= '</section>';
	} else {
		$out = '<section class="aeo-overview">';
		if ( $title !== '' ) {
			$out .= '<h' . $h . ' class="aeo-overview-heading">' . esc_html( $title ) . '</h' . $h . '>';
		}
		$out .= '<p class="aeo-overview-text">' . esc_html( $overview ) . '</p>';
		$out .= '</section>';
	}
	return apply_filters( 'aeo_shortcode_overview_markup', $out, $overview, $args );
}

/**
 * Build FAQs markup.
 */
function aeo_build_faqs_markup( $faqs, $args ) {
	if ( empty( $faqs ) ) { return ''; }
	$limit = $args['limit'];
	if ( $limit > 0 && count( $faqs ) > $limit ) { $faqs = array_slice( $faqs, 0, $limit ); }

	$title = trim( (string) $args['title_faqs'] );
	$h = intval( $args['heading_faqs'] );
	$accordion = isset( $args['faq_accordion'] ) ? (bool) $args['faq_accordion'] : $args['accordion'];
	$list_style = $args['list_style'];
	$uid_base = uniqid( 'aeo_faq_', true );

	$attr_accordion = $accordion ? ' data-accordion="true"' : '';
	$out = '<section class="aeo-faqs" aria-label="' . esc_attr__( 'FAQs', 'anslift' ) . '"' . $attr_accordion . '>';
	if ( $title !== '' ) {
		$out .= '<h' . $h . ' class="aeo-faqs-heading">' . esc_html( $title ) . '</h' . $h . '>';
	}

	if ( $accordion ) {
		// Use div container of items for accordion consistency regardless of list_style.
		$out .= '<div class="aeo-faq-items">';
		$index = 0;
		foreach ( $faqs as $faq ) {
			$index++;
			$q_id = $uid_base . '_' . $index;
			$out .= '<div class="aeo-faq-item aeo-accordion-item">';
			$out .= '<button type="button" class="aeo-faq-question-button aeo-accordion-trigger" aria-expanded="false" aria-controls="' . esc_attr( $q_id ) . '">' . esc_html( $faq['question'] ) . '</button>';
			$out .= '<div id="' . esc_attr( $q_id ) . '" class="aeo-faq-answer aeo-accordion-panel" hidden>' . esc_html( $faq['answer'] ) . '</div>';
			$out .= '</div>';
		}
		$out .= '</div>';
	} else {
		if ( 'dl' === $list_style ) {
			$out .= '<dl class="aeo-faq-list">';
			foreach ( $faqs as $faq ) {
				$out .= '<dt class="aeo-faq-question">' . esc_html( $faq['question'] ) . '</dt>';
				$out .= '<dd class="aeo-faq-answer">' . esc_html( $faq['answer'] ) . '</dd>';
			}
			$out .= '</dl>';
		} elseif ( in_array( $list_style, array( 'ul', 'ol' ), true ) ) {
			$out .= '<' . $list_style . ' class="aeo-faq-list">';
			foreach ( $faqs as $faq ) {
				$out .= '<li class="aeo-faq-item"><span class="aeo-faq-question">' . esc_html( $faq['question'] ) . '</span><div class="aeo-faq-answer">' . esc_html( $faq['answer'] ) . '</div></li>';
			}
			$out .= '</' . $list_style . '>';
		} else { // div
			$out .= '<div class="aeo-faq-list">';
			foreach ( $faqs as $faq ) {
				$out .= '<div class="aeo-faq-item"><p class="aeo-faq-question"><strong>' . esc_html( $faq['question'] ) . '</strong></p><p class="aeo-faq-answer">' . esc_html( $faq['answer'] ) . '</p></div>';
			}
			$out .= '</div>';
		}
	}
	$out .= '</section>';
	return apply_filters( 'aeo_shortcode_faqs_markup', $out, $faqs, $args );
}

/**
 * Build (optional) schema object for FAQPage subset when shortcode is explicitly asked to do so.
 */
function aeo_build_faq_schema_object( $faqs ) {
	$questions = array();
	$clean_text = function_exists( 'aeo_schema_clean_text' ) ? 'aeo_schema_clean_text' : 'sanitize_text_field';
	foreach ( $faqs as $faq ) {
		if ( ! is_array( $faq ) ) {
			continue;
		}
		$question = call_user_func( $clean_text, isset( $faq['question'] ) ? $faq['question'] : '' );
		$answer   = call_user_func( $clean_text, isset( $faq['answer'] ) ? $faq['answer'] : '' );
		if ( '' === $question || '' === $answer ) {
			continue;
		}
		$questions[] = array(
			'@type' => 'Question',
			'name' => $question,
			'acceptedAnswer' => array(
				'@type' => 'Answer',
				'text' => $answer,
			),
		);
	}
	if ( empty( $questions ) ) {
		return array();
	}
	return array(
		'@context' => 'https://schema.org',
		'@type' => 'FAQPage',
		'mainEntity' => $questions,
	);
}

/**
 * Shortcode callback.
 */
function aeo_shortcode_render_content( $atts = array(), $content = '' ) { // phpcs:ignore
	$args = aeo_normalize_args( $atts );
	$post = null;
	if ( $args['post_id'] > 0 ) {
		$post = get_post( $args['post_id'] );
	} else {
		global $post; // use global in loop
	}
	if ( ! $post instanceof WP_Post ) { return ''; }
	if ( post_password_required( $post ) ) { return ''; }

	// Suppress sections that are auto-inserted on singular views for the same post.
	$auto_summary = false;
	$auto_faqs = false;
	$context_post_id = aeo_get_auto_insert_context_post_id( $post );
	if ( $context_post_id && (int) $context_post_id === (int) $post->ID && aeo_is_auto_insert_supported_post_type( $context_post_id ) ) {
		$auto_summary = (int) get_option( 'aeo_auto_insert_summary', 1 ) === 1;
		$auto_faqs    = (int) get_option( 'aeo_auto_insert_faqs', 1 ) === 1;
	}

	// Fragment cache key (simple). Exclude volatile show_schema (schema appended outside core fragment) and class.
	$cache_key = '';
	if ( $args['cache'] > 0 ) {
		$cache_fingerprint = md5( wp_json_encode( array(
			'post' => $post->ID,
			'show' => $args['show'],
			'limit' => $args['limit'],
			'list_style' => $args['list_style'],
			'accordion' => $args['accordion'],
			'overview_accordion' => $args['overview_accordion'],
			'faq_accordion' => $args['faq_accordion'],
			'empty_behavior' => $args['empty_behavior'],
		) ) );
		$cache_key = 'aeo_sc_' . $cache_fingerprint;
		$cached = get_transient( $cache_key );
		if ( false !== $cached ) {
			return $cached; // Already sanitized markup.
		}
	}

	$meta = aeo_get_content_meta( $post->ID );
	$overview = $meta['overview'];
	$faqs = $meta['faqs'];

	$show_overview = in_array( $args['show'], array( 'overview', 'both' ), true );
	$show_faqs = in_array( $args['show'], array( 'faqs', 'both' ), true );

	// Suppress sections that are auto-inserted
	if ( $auto_summary && $show_overview ) {
		$show_overview = false;
	}
	if ( $auto_faqs && $show_faqs ) {
		$show_faqs = false;
	}

	$has_overview = $show_overview && '' !== trim( (string) $overview );
	$has_faqs = $show_faqs && ! empty( $faqs );

	if ( ! $has_overview && ! $has_faqs ) {
		if ( 'message' === $args['empty_behavior'] ) {
			return '<div class="aeo-content aeo-empty">' . esc_html__( 'No AI enhancement available yet.', 'anslift' ) . '</div>';
		} elseif ( 'placeholder' === $args['empty_behavior'] ) {
			return '<div class="aeo-content aeo-placeholder"></div>';
		}
		return '';
	}

	$pieces = array();
	if ( $has_overview ) {
		$pieces[] = aeo_build_overview_markup( $overview, $args );
	}
	if ( $has_faqs ) {
		$pieces[] = aeo_build_faqs_markup( $faqs, $args );
	}

	$classes = array( 'aeo-content', 'aeo-mode-' . esc_attr( $args['show'] ) );
	if ( $args['class'] ) { $classes[] = $args['class']; }
	$needs_js = ( $args['overview_accordion'] || $args['faq_accordion'] || $args['accordion'] );

	// Enqueue assets now (first render pass) to avoid duplication.
	aeo_enqueue_frontend_assets( $needs_js );

	$html = '<div class="' . implode( ' ', array_map( 'sanitize_html_class', $classes ) ) . '">' . implode( '', $pieces ) . '</div>';

	// Optional schema injection (FAQ only) if allowed by constant and requested.
	static $schema_printed = false;
	$faq_schema_already_printed = function_exists( 'aeo_schema_is_faq_printed' ) ? aeo_schema_is_faq_printed( $post->ID ) : false;
	if ( $args['show_schema'] && AEO_ALLOW_SHORTCODE_SCHEMA && ! $schema_printed && $has_faqs && ! $faq_schema_already_printed ) {
		$schema_obj = aeo_build_faq_schema_object( $faqs );
		$schema_obj = apply_filters( 'aeo_shortcode_schema_object', $schema_obj, $faqs, $args );
		if ( is_array( $schema_obj ) && ! empty( $schema_obj ) ) {
			$html .= '<script type="application/ld+json">' . wp_json_encode( $schema_obj, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . '</script>';
			$schema_printed = true;
			if ( function_exists( 'aeo_schema_mark_faq_printed' ) ) {
				aeo_schema_mark_faq_printed( $post->ID );
			}
		}
	}

	$html = apply_filters( 'aeo_shortcode_markup', $html, $args, $post );
	do_action( 'aeo_shortcode_after_render', $args, $post->ID, array( 'has_overview' => $has_overview, 'has_faqs' => $has_faqs ) );

	if ( $cache_key && $args['cache'] > 0 ) {
		set_transient( $cache_key, $html, $args['cache'] );
	}
	return $html;
}

/**
 * Template tag: return markup.
 */
function aeo_render_content( $args = array() ) {
	return aeo_shortcode_render_content( $args );
}

/**
 * Template tag: echo markup.
 */
function aeo_the_content( $args = array() ) {
	echo aeo_render_content( $args ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- already escaped in renderer
}
