<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Minimal POT generator for WordPress-style i18n calls.
 * Not as complete as WP-CLI make-pot, but good enough for small plugins.
 *
 * Usage: php bin/make-pot.php
 */

$root = realpath(__DIR__ . '/..');
$pluginSlug = 'anslift';
$languagesDir = $root . '/languages';
$outputFile = $languagesDir . '/' . $pluginSlug . '.pot';

$includePatterns = [
    '/\.php$/i',
];
$excludeDirs = [
    'supbase-backend', 'vendor', 'node_modules', 'assets', '.git', '.vscode', 'public/assets', 'admin/assets'
];

$functions = [
    '__', '_e', '_x', '_ex', '_n', '_nx',
    'esc_html__', 'esc_attr__', 'esc_html_e', 'esc_attr_e'
];

$files = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS));
$potEntries = [];

function isExcluded($path, $excludeDirs) {
    foreach ($excludeDirs as $dir) {
        if (strpos($path, DIRECTORY_SEPARATOR . $dir . DIRECTORY_SEPARATOR) !== false) {
            return true;
        }
    }
    return false;
}

foreach ($files as $file) {
    $path = $file->getRealPath();
    if (!preg_match($includePatterns[0], $path)) continue;
    if (isExcluded($path, $excludeDirs)) continue;

    $code = file_get_contents($path);
    if ($code === false) continue;

    // Basic regex for gettext calls: __( 'text', 'domain' ) or "text", 'domain'
    $pattern = '/\b(' . implode('|', array_map('preg_quote', $functions)) . ')\s*\(\s*([\'\"][\s\S]*?[\'\"])\s*,\s*([\'\"][^\'\"]+[\'\"])\s*\)/m';
    if (preg_match_all($pattern, $code, $matches, PREG_OFFSET_CAPTURE)) {
        foreach ($matches[0] as $i => $_) {
            $func = $matches[1][$i][0];
            $textRaw = $matches[2][$i][0];
            $domainRaw = $matches[3][$i][0];

            $text = stripcslashes(substr($textRaw, 1, -1));
            $domain = stripcslashes(substr($domainRaw, 1, -1));
            if ($domain !== $pluginSlug) continue; // only capture our domain

            // Determine line number
            $before = substr($code, 0, $matches[0][$i][1]);
            $line = substr_count($before, "\n") + 1;

            $relPath = str_replace($root . DIRECTORY_SEPARATOR, '', $path);
            $key = $text . "\0" . $relPath . ':' . $line;
            $potEntries[$key] = [
                'file' => $relPath,
                'line' => $line,
                'text' => $text,
            ];
        }
    }
}

// Write POT
$header =
    "msgid \"\"\n" .
    "msgstr \"\"\n" .
    "\"Project-Id-Version: $pluginSlug\\n\"\n" .
    "\"Report-Msgid-Bugs-To: \\n\"\n" .
    "\"POT-Creation-Date: " . gmdate('Y-m-d H:iO') . "\\n\"\n\n\n";

$out = $header . "\n";

foreach ($potEntries as $entry) {
    $out .= "#: {$entry['file']}:{$entry['line']}\n";
    $msgid = addcslashes($entry['text'], "\n\r\t\"\\");
    $out .= "msgid \"$msgid\"\n";
    $out .= "msgstr \"\"\n\n";
}

// Ensure languages directory exists; prefer WP_Filesystem when available
if (!is_dir($languagesDir)) {
    if (defined('ABSPATH') && file_exists(ABSPATH . 'wp-admin/includes/file.php')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        global $wp_filesystem;
        if (! $wp_filesystem) {
            WP_Filesystem();
        }
        if ($wp_filesystem && method_exists($wp_filesystem, 'mkdir')) {
            $wp_filesystem->mkdir($languagesDir);
        } else {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
            mkdir($languagesDir, 0777, true);
        }
    } else {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
        mkdir($languagesDir, 0777, true);
    }
}

// Write POT file; prefer WP_Filesystem when available
$writeOk = false;
if (defined('ABSPATH') && file_exists(ABSPATH . 'wp-admin/includes/file.php')) {
    require_once ABSPATH . 'wp-admin/includes/file.php';
    global $wp_filesystem;
    if (! $wp_filesystem) {
        WP_Filesystem();
    }
    if ($wp_filesystem && method_exists($wp_filesystem, 'put_contents')) {
        if (! $wp_filesystem->is_dir($languagesDir)) {
            $wp_filesystem->mkdir($languagesDir);
        }
        $writeOk = (bool) $wp_filesystem->put_contents($outputFile, $out, FS_CHMOD_FILE);
    }
}

if (! $writeOk) {
    // Fallback for non-WordPress CLI use
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
    $writeOk = (file_put_contents($outputFile, $out) !== false);
}

// Provide a safe esc_html fallback if WordPress is not loaded (CLI usage)
if ( ! function_exists( 'esc_html' ) ) {
    function esc_html( $text ) {
        return htmlspecialchars( (string) $text, ENT_QUOTES, 'UTF-8' );
    }
}

if ( $writeOk ) {
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped via esc_html above
    echo 'Generated: ' . esc_html( $outputFile ) . \PHP_EOL;
} else {
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped via esc_html above
    echo 'Failed to write: ' . esc_html( $outputFile ) . \PHP_EOL;
}
