<?php
// admin/menu.php

if (!defined('WPINC')) {
    die;
}

if ( ! function_exists( 'aeo_get_upgrade_url' ) ) {
    function aeo_get_upgrade_url() {
        $default = 'https://aeogeorank.com/';
        $url = apply_filters( 'aeo_upgrade_url', $default );
        $url = wp_http_validate_url( $url );
        if ( ! $url ) {
            $url = $default;
        }
        return $url;
    }
}

function aeo_allow_upgrade_redirect_host( $hosts ) {
    $target = aeo_get_upgrade_url();
    $host = $target ? wp_parse_url( $target, PHP_URL_HOST ) : '';
    if ( $host ) {
        $hosts[] = $host;
    }
    return array_unique( $hosts );
}

function aeo_plugin_add_admin_menu() {
    // Back-compat redirect for old slug
    if ( isset( $_GET['page'] ) && 'aeo-content-plugin' === sanitize_key( wp_unslash( $_GET['page'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        wp_safe_redirect( admin_url('admin.php?page=anslift') );
        exit;
    }
    add_menu_page('Anslift', 'Anslift', 'manage_options', 'anslift', 'aeo_plugin_admin_page_content', 'dashicons-superhero-alt', 25);
    add_submenu_page(
        'anslift',
        __( 'Upgrade', 'anslift' ),
        __( 'Upgrade', 'anslift' ),
        'manage_options',
        'anslift-upgrade',
        'aeo_plugin_upgrade_redirect'
    );
}
add_action('admin_menu', 'aeo_plugin_add_admin_menu');

function aeo_plugin_admin_menu_customizations() {
    $upgrade_url = aeo_get_upgrade_url();
    if ( ! $upgrade_url ) {
        return;
    }
    $ajax_url = admin_url( 'admin-ajax.php' );
    $status_nonce = wp_create_nonce( 'aeo_status_nonce' );
    ?>
    <script>
    document.addEventListener('DOMContentLoaded', function () {
        var submenu = document.querySelector('#toplevel_page_anslift ul.wp-submenu');
        if (!submenu) {
            return;
        }
        var settingsLink = submenu.querySelector('a[href$="page=anslift"]');
        if (settingsLink && settingsLink.parentElement) {
            settingsLink.parentElement.style.display = 'none';
        }
        var upgradeLink = submenu.querySelector('a[href$="page=anslift-upgrade"]');
        if (upgradeLink) {
            upgradeLink.href = '<?php echo esc_js( $upgrade_url ); ?>';
            upgradeLink.target = '_blank';
            upgradeLink.rel = 'noopener noreferrer';
        }

        if (!upgradeLink || !upgradeLink.parentElement) {
            return;
        }

        function hideUpgradeMenu() {
            upgradeLink.parentElement.style.display = 'none';
        }

        function readPlanCache() {
            try {
                var raw = localStorage.getItem('aeo_plan_cache');
                if (!raw) { return null; }
                var parsed = JSON.parse(raw);
                if (!parsed || !parsed.plan) { return null; }
                return parsed;
            } catch (e) { return null; }
        }

        function writePlanCache(plan) {
            try {
                localStorage.setItem('aeo_plan_cache', JSON.stringify({ plan: plan, ts: Date.now() }));
            } catch (e) { /* noop */ }
        }

        function applyPlan(plan) {
            if (String(plan).toLowerCase() === 'pro') {
                hideUpgradeMenu();
            }
        }

        var cached = readPlanCache();
        if (cached && cached.plan && cached.ts && (Date.now() - cached.ts) < 300000) {
            applyPlan(cached.plan);
        } else if (window.fetch) {
            var params = new URLSearchParams();
            params.append('action', 'aeo_usage_status');
            params.append('nonce', '<?php echo esc_js( $status_nonce ); ?>');
            fetch('<?php echo esc_js( $ajax_url ); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
                body: params.toString(),
                credentials: 'same-origin'
            }).then(function (r) { return r.json(); }).then(function (resp) {
                if (resp && resp.success && resp.data && resp.data.plan) {
                    writePlanCache(resp.data.plan);
                    applyPlan(resp.data.plan);
                }
            }).catch(function () { /* noop */ });
        }
    });
    </script>
    <?php
}
add_action( 'admin_head', 'aeo_plugin_admin_menu_customizations' );

function aeo_plugin_upgrade_redirect() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'anslift' ) );
    }
    $target = aeo_get_upgrade_url();
    if ( ! $target ) {
        wp_die( esc_html__( 'Invalid upgrade destination.', 'anslift' ) );
    }
    add_filter( 'allowed_redirect_hosts', 'aeo_allow_upgrade_redirect_host' );
    wp_safe_redirect( $target );
    exit;
}

function aeo_plugin_admin_page_content() {
    $license_key = get_option('aeo_license_key', '');
    $license_status = get_option('aeo_license_status', 'inactive');
    $license_error = get_option('aeo_license_error', '');
    $auto_insert_summary = (int) get_option('aeo_auto_insert_summary', 1);
    $auto_insert_faqs = (int) get_option('aeo_auto_insert_faqs', 1);
    $auto_analyze = (int) get_option('aeo_auto_analyze', 1); // new option (defaults enabled per requirements)
    $upgrade_url = aeo_get_upgrade_url();
    $upgrade_label = __( 'Upgrade Plan', 'anslift' );
    $cancel_note = __( 'To cancel, use the link in the subscription email sent to the address you used at checkout.', 'anslift' );

    // Scripts and localized data are enqueued via admin_enqueue_scripts in analysis.php
    ?>
    <div class="wrap anslift-wrap">
        <header class="anslift-header">
            <h1 class="anslift-title"><?php echo esc_html(get_admin_page_title()); ?></h1>
            <p class="anslift-subtitle"><?php esc_html_e('Manage your AEO license and view analyzed posts.', 'anslift'); ?></p>
        </header>

        <div class="anslift-dashboard-grid">
            <!-- License Panel -->
            <div id="aeo-license-panel" class="anslift-card">
                <div class="anslift-card-header">
                    <h2><?php esc_html_e('License Settings', 'anslift'); ?></h2>
                </div>
                <div class="anslift-card-body">
                    <!-- Usage status (plan pill + usage bar) -->
                    <div id="aeo-usage-status" class="aeo-usage-status" aria-live="polite">
                        <div class="aeo-usage-header">
                            <span class="aeo-plan-pill" id="aeo-plan-pill" aria-label="Plan"></span>
                            <button type="button" class="aeo-usage-tooltip" aria-label="Usage info" title="Auto-analysis counts toward your Pro quota. Pro quota resets each billing period.">
                                <span class="dashicons dashicons-info"></span>
                            </button>
                        </div>
                        <div class="aeo-usage-line" id="aeo-usage-line">&nbsp;</div>
                        <div class="aeo-usage-bar" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0" aria-label="Usage progress">
                            <div class="aeo-usage-fill" style="width:0%"></div>
                        </div>
                        <div class="aeo-usage-fallback" id="aeo-usage-fallback" hidden>Unable to fetch usage</div>
                    </div>
                    
                    <div class="aeo-upgrade-actions">
                        <a class="button button-primary anslift-btn-block" href="<?php echo esc_url( $upgrade_url ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html( $upgrade_label ); ?></a>
                    </div>
                    <p class="anslift-text-muted aeo-subscription-note" hidden><?php echo esc_html( $cancel_note ); ?></p>

                    <div class="anslift-license-form">
                        <?php if ($license_status === 'active') : ?>
                            <div class="anslift-status-message success">
                                <span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Your license is active.', 'anslift'); ?>
                            </div>
                            <div class="anslift-form-group">
                                <label><?php esc_html_e('License Key', 'anslift'); ?></label>
                                <div class="anslift-input-group">
                                    <input type="text" class="regular-text" value="************<?php echo esc_attr(substr($license_key, -4)); ?>" readonly disabled />
                                    <button id="aeo-deactivate-license" class="button button-secondary"><?php esc_html_e('Deactivate', 'anslift'); ?></button>
                                </div>
                            </div>
                        <?php else : ?>
                            <p class="anslift-text-muted"><?php esc_html_e('Please enter your license key to enable Pro features and unlimited analysis.', 'anslift'); ?></p>
                            <?php if (!empty($license_error)) : ?>
                                <div class="anslift-status-message error">
                                    <strong><?php esc_html_e('Activation Error:', 'anslift'); ?></strong> <?php echo esc_html($license_error); ?>
                                </div>
                            <?php endif; ?>
                            <div class="anslift-form-group">
                                <label for="aeo_license_key_input"><?php esc_html_e('License Key', 'anslift'); ?></label>
                                <div class="anslift-input-group">
                                    <input type="text" id="aeo_license_key_input" class="regular-text" placeholder="Enter your license key" />
                                    <button id="aeo-activate-license" class="button button-primary"><?php esc_html_e('Activate', 'anslift'); ?></button>
                                </div>
                            </div>
                        <?php endif; ?>
                        <div id="aeo-license-spinner" class="spinner"></div>
                    </div>
                </div>
            </div>

            <!-- Settings Panel -->
            <div id="aeo-display-settings" class="anslift-card">
                <div class="anslift-card-header">
                    <h2><?php esc_html_e('Display & Automation', 'anslift'); ?></h2>
                </div>
                <div class="anslift-card-body">
                    <p class="anslift-text-muted"><?php esc_html_e('Control front-end insertion and automatic AI analysis of newly published posts.', 'anslift'); ?></p>
                    <p class="anslift-text-muted" style="margin-bottom: 20px;">
                        <?php esc_html_e('Choose which AI sections to inject automatically on single posts.', 'anslift'); ?>
                    </p>

                    <div class="anslift-settings-list">
                        <!-- Setting Item -->
                        <div class="anslift-setting-item">
                            <div class="anslift-setting-content">
                                <label for="aeo_auto_insert_summary_toggle" class="anslift-setting-label"><?php esc_html_e('Insert AI Summary', 'anslift'); ?></label>
                                <p class="anslift-setting-desc"><?php esc_html_e('Automatically insert AI summary before content.', 'anslift'); ?></p>
                            </div>
                            <div class="anslift-toggle-wrapper">
                                <label class="anslift-switch">
                                    <input type="checkbox" id="aeo_auto_insert_summary_toggle" value="1" <?php checked( $auto_insert_summary, 1 ); ?> />
                                    <span class="anslift-slider round"></span>
                                </label>
                            </div>
                        </div>

                        <!-- Setting Item -->
                        <div class="anslift-setting-item">
                            <div class="anslift-setting-content">
                                <label for="aeo_auto_insert_faqs_toggle" class="anslift-setting-label"><?php esc_html_e('Insert FAQs', 'anslift'); ?></label>
                                <p class="anslift-setting-desc"><?php esc_html_e('Automatically insert FAQs after content.', 'anslift'); ?></p>
                            </div>
                            <div class="anslift-toggle-wrapper">
                                <label class="anslift-switch">
                                    <input type="checkbox" id="aeo_auto_insert_faqs_toggle" value="1" <?php checked( $auto_insert_faqs, 1 ); ?> />
                                    <span class="anslift-slider round"></span>
                                </label>
                            </div>
                        </div>

                        <!-- Setting Item -->
                        <div class="anslift-setting-item <?php echo $license_status !== 'active' ? 'anslift-disabled' : ''; ?>">
                            <div class="anslift-setting-content">
                                <label for="aeo_auto_analyze_toggle" class="anslift-setting-label"><?php esc_html_e('Auto-Analyze New Posts', 'anslift'); ?></label>
                                <p class="anslift-setting-desc">
                                    <?php
                                    if ( $license_status !== 'active' ) {
                                        esc_html_e('Upgrade to enable automatic analysis.', 'anslift');
                                    } else {
                                        esc_html_e('Analyze newly published posts automatically.', 'anslift');
                                    }
                                    ?>
                                </p>
                            </div>
                            <div class="anslift-toggle-wrapper">
                                <label class="anslift-switch">
                                    <input type="checkbox" id="aeo_auto_analyze_toggle" value="1" <?php checked( $auto_analyze, 1 ); ?> <?php disabled( $license_status !== 'active' ); ?> />
                                    <span class="anslift-slider round"></span>
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <div class="anslift-status-bar">
                        <div id="aeo-auto-insert-spinner" class="spinner"></div>
                        <div id="aeo-auto-insert-status"></div>
                    </div>

                    <script>
                    (function($){
                        $(function(){
                            var $summaryToggle = $('#aeo_auto_insert_summary_toggle');
                            var $faqToggle = $('#aeo_auto_insert_faqs_toggle');
                            var $autoAnalyze = $('#aeo_auto_analyze_toggle');
                            var $spinner = $('#aeo-auto-insert-spinner');
                            var $status = $('#aeo-auto-insert-status');
                            function bindAutoInsertToggle($toggle, action) {
                                if (!$toggle.length) { return; }
                                $toggle.on('change', function(){
                                    $spinner.addClass('is-active');
                                    $status.text('');
                                    $.post(ajaxurl, {
                                        action: action,
                                        nonce: (window.aeo_ajax_object && window.aeo_ajax_object.settings_nonce) ? window.aeo_ajax_object.settings_nonce : '',
                                        enable: $toggle.is(':checked') ? 1 : 0
                                    }, function(resp){
                                        $spinner.removeClass('is-active');
                                        if(resp && resp.success){
                                            $status.css('color','#2271b1').text(resp.data.message || 'Saved');
                                        } else {
                                            $status.css('color','#b32d2e').text((resp && resp.data && resp.data.message) ? resp.data.message : 'Error saving');
                                        }
                                    }).fail(function(){
                                        $spinner.removeClass('is-active');
                                        $status.css('color','#b32d2e').text('Request failed');
                                    });
                                });
                            }

                            bindAutoInsertToggle($summaryToggle, 'aeo_toggle_auto_insert_summary');
                            bindAutoInsertToggle($faqToggle, 'aeo_toggle_auto_insert_faqs');

                            $autoAnalyze.on('change', function(){
                                $spinner.addClass('is-active');
                                $status.text('');
                                $.post(ajaxurl, {
                                    action: 'aeo_toggle_auto_analyze',
                                    nonce: (window.aeo_ajax_object && window.aeo_ajax_object.settings_nonce) ? window.aeo_ajax_object.settings_nonce : '',
                                    enable: $autoAnalyze.is(':checked') ? 1 : 0
                                }, function(resp){
                                    $spinner.removeClass('is-active');
                                    if(resp && resp.success){
                                        $status.css('color','#2271b1').text(resp.data.message || 'Saved');
                                    } else {
                                        $status.css('color','#b32d2e').text((resp && resp.data && resp.data.message) ? resp.data.message : 'Error saving');
                                    }
                                }).fail(function(){
                                    $spinner.removeClass('is-active');
                                    $status.css('color','#b32d2e').text('Request failed');
                                });
                            });
                        });
                    })(jQuery);
                    </script>
                </div>
            </div>
        </div>

        <div class="anslift-card anslift-full-width">
            <div class="anslift-card-header">
                <h2><?php esc_html_e('Analyzed Posts', 'anslift'); ?></h2>
            </div>
            <div class="anslift-card-body">
            <?php
            // Pagination setup (index-driven to avoid slow DB filters)
            $per_page = 10; // fixed per requirements
            $paged = isset( $_GET['paged'] ) ? max( 1, intval( sanitize_text_field( wp_unslash( $_GET['paged'] ) ) ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $index = function_exists( 'aeo_get_analyzed_index' ) ? aeo_get_analyzed_index() : array();
            $analyzed_ids = array_keys( $index );
            $total_items  = count( $analyzed_ids );
            $total_pages  = $total_items > 0 ? (int) ceil( $total_items / $per_page ) : 0;
            if ( $total_pages > 0 ) {
                $paged = min( $paged, $total_pages );
            } else {
                $paged = 1;
            }
            $offset       = ( $paged - 1 ) * $per_page;
            $page_ids     = array_slice( $analyzed_ids, $offset, $per_page );

            if ( ! empty( $page_ids ) ) {
                $analyzed_posts = new WP_Query( array(
                    'post_type'              => 'post',
                    'post_status'            => 'any',
                    'posts_per_page'         => count( $page_ids ),
                    'post__in'               => array_map( 'intval', $page_ids ),
                    'orderby'                => 'post__in',
                    'order'                  => 'ASC',
                    'no_found_rows'          => true,
                    'update_post_term_cache' => false,
                    'ignore_sticky_posts'    => true,
                ) );
            } else {
                $analyzed_posts = new WP_Query( array( 'post_type' => 'post', 'post__in' => array( 0 ), 'posts_per_page' => 0, 'no_found_rows' => true ) );
            }

            // Build base URL for pagination links (preserve other query args except paged)
            $base_url = remove_query_arg( 'paged' );
            $page_links = '';
            if ( $total_pages > 1 ) {
                $page_links = paginate_links( array(
                    'base'      => add_query_arg( 'paged', '%#%', $base_url ),
                    'format'    => '',
                    'prev_text' => '&laquo;',
                    'next_text' => '&raquo;',
                    'total'     => $total_pages,
                    'current'   => $paged,
                    'type'      => 'array',
                ) );
            }

            // Enhanced pagination builder (top) removed per user request
            ?>
            <table class="wp-list-table widefat fixed striped anslift-table">
                <thead>
                    <tr>
                        <th scope="col"><?php esc_html_e( 'Post Title', 'anslift' ); ?></th>
                        <th scope="col"><?php esc_html_e( 'AI Summary', 'anslift' ); ?></th>
                        <th scope="col" style="width: 15%;"><?php esc_html_e( 'FAQs Generated', 'anslift' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php
                if ( $analyzed_posts->have_posts() ) :
                    while ( $analyzed_posts->have_posts() ) : $analyzed_posts->the_post();
                        $overview = get_post_meta( get_the_ID(), '_aeo_ai_overview', true );
                        $faqs     = get_post_meta( get_the_ID(), '_aeo_faqs', true );
                        ?>
                        <tr>
                            <td>
                                <span class="aeo-post-title"><?php echo esc_html( get_the_title() ); ?></span>
                                <div class="aeo-post-actions">
                                    <a href="<?php echo esc_url( get_permalink() ); ?>" class="button button-small aeo-view-post" aria-label="<?php echo esc_attr__( 'View post', 'anslift' ); ?>"><?php echo esc_html__( 'View', 'anslift' ); ?></a>
                                    <a href="<?php echo esc_url( get_edit_post_link() ); ?>" class="button button-small aeo-edit-post" aria-label="<?php echo esc_attr__( 'Edit post', 'anslift' ); ?>"><?php echo esc_html__( 'Edit', 'anslift' ); ?></a>
                                </div>
                            </td>
                            <td><?php echo esc_html( wp_trim_words( $overview, 20, '...' ) ); ?></td>
                            <td><span class="anslift-badge"><?php echo esc_html( is_array( $faqs ) ? count( $faqs ) : 0 ); ?></span></td>
                        </tr>
                        <?php
                    endwhile;
                    wp_reset_postdata();
                else : ?>
                    <tr>
                        <td colspan="3"><?php esc_html_e( 'No content has been generated yet.', 'anslift' ); ?></td>
                    </tr>
                <?php endif; ?>
                </tbody>
            </table>
            <?php
            // Enhanced pagination builder (bottom)
            if ( $page_links ) {
                echo wp_kses_post( aeo_render_pagination_nav( $page_links, $paged, $total_pages, $base_url, $total_items, 'bottom' ) );
            }
            ?>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Render pagination navigation HTML for analyzed posts.
 */
function aeo_render_pagination_nav( $page_links, $paged, $total_pages, $base_url, $total_items, $position = 'top' ) {
    $first_link = '';
    $last_link  = '';
    if ( $paged > 1 ) {
    $first_link = '<a class="first-page" href="' . esc_url( add_query_arg( 'paged', 1, $base_url ) ) . '" aria-label="' . esc_attr__( 'First page', 'anslift' ) . '">&laquo;</a>';
    } else {
        $first_link = '<span class="tablenav-pages-navspan first-page disabled" aria-hidden="true">&laquo;</span>';
    }
    if ( $paged < $total_pages ) {
    $last_link = '<a class="last-page" href="' . esc_url( add_query_arg( 'paged', $total_pages, $base_url ) ) . '" aria-label="' . esc_attr__( 'Last page', 'anslift' ) . '">&raquo;</a>';
    } else {
        $last_link = '<span class="tablenav-pages-navspan last-page disabled" aria-hidden="true">&raquo;</span>';
    }
    $page_nav = $first_link . ' ' . implode( ' ', array_map( 'wp_kses_post', $page_links ) ) . ' ' . $last_link;
    ob_start();
    ?>
    <nav class="tablenav <?php echo esc_attr( $position ); ?> aeo-pagination" aria-label="<?php echo esc_attr__( 'Analyzed posts pagination', 'anslift' ); ?>">
        <div class="tablenav-pages">
            <span class="displaying-num"><?php
                /* translators: %s: number of items */
                echo esc_html( sprintf( _n( '%s item', '%s items', $total_items, 'anslift' ), number_format_i18n( $total_items ) ) );
            ?></span>
            <span class="pagination-status"><?php
                /* translators: 1: current page number, 2: total number of pages */
                echo esc_html( sprintf( __( 'Page %1$d of %2$d', 'anslift' ), $paged, $total_pages ) );
            ?></span>
            <span class="pagination-links"><?php echo wp_kses_post( $page_nav ); ?></span>
        </div>
    </nav>
    <?php
    return ob_get_clean();
}

// AJAX handler for auto insert toggle
function aeo_toggle_auto_insert_summary() {
    check_ajax_referer('aeo_settings_nonce', 'nonce');
    if ( ! current_user_can('manage_options') ) {
        wp_send_json_error(['message' => __('Permission denied', 'anslift')]);
    }
    $enable = isset($_POST['enable']) ? (int) $_POST['enable'] : 0;
    update_option('aeo_auto_insert_summary', $enable ? 1 : 0);

    if ( function_exists( 'aeo_sync_legacy_auto_insert_option' ) ) {
        aeo_sync_legacy_auto_insert_option();
    }

    wp_send_json_success(['message' => $enable ? __('Automatic summary insertion enabled.', 'anslift') : __('Automatic summary insertion disabled.', 'anslift')]);
}
add_action('wp_ajax_aeo_toggle_auto_insert_summary', 'aeo_toggle_auto_insert_summary');

function aeo_toggle_auto_insert_faqs() {
    check_ajax_referer('aeo_settings_nonce', 'nonce');
    if ( ! current_user_can('manage_options') ) {
        wp_send_json_error(['message' => __('Permission denied', 'anslift')]);
    }
    $enable = isset($_POST['enable']) ? (int) $_POST['enable'] : 0;
    update_option('aeo_auto_insert_faqs', $enable ? 1 : 0);

    if ( function_exists( 'aeo_sync_legacy_auto_insert_option' ) ) {
        aeo_sync_legacy_auto_insert_option();
    }

    wp_send_json_success(['message' => $enable ? __('Automatic FAQ insertion enabled.', 'anslift') : __('Automatic FAQ insertion disabled.', 'anslift')]);
}
add_action('wp_ajax_aeo_toggle_auto_insert_faqs', 'aeo_toggle_auto_insert_faqs');

// Legacy handler retained for backward compatibility with cached admin assets.
function aeo_toggle_auto_insert() {
    check_ajax_referer('aeo_settings_nonce', 'nonce');
    if ( ! current_user_can('manage_options') ) {
        wp_send_json_error(['message' => __('Permission denied', 'anslift')]);
    }
    $enable = isset($_POST['enable']) ? (int) $_POST['enable'] : 0;
    update_option('aeo_auto_insert_summary', $enable ? 1 : 0);
    update_option('aeo_auto_insert_faqs', $enable ? 1 : 0);

    if ( function_exists( 'aeo_sync_legacy_auto_insert_option' ) ) {
        aeo_sync_legacy_auto_insert_option();
    }

    wp_send_json_success(['message' => $enable ? __('Auto insertion enabled.', 'anslift') : __('Auto insertion disabled.', 'anslift')]);
}
add_action('wp_ajax_aeo_toggle_auto_insert', 'aeo_toggle_auto_insert');

// AJAX handler for auto analyze toggle (Pro only)
function aeo_toggle_auto_analyze() {
    check_ajax_referer('aeo_settings_nonce', 'nonce');
    if ( ! current_user_can('manage_options') ) {
        wp_send_json_error(['message' => __('Permission denied', 'anslift')]);
    }
    // Only allow enabling if license active; disabling is always allowed
    $enable = isset($_POST['enable']) ? (int) $_POST['enable'] : 0;
    $license_status = get_option('aeo_license_status');
    if ( $enable && $license_status !== 'active' ) {
        wp_send_json_error(['message' => __('Activate your license to enable automatic analysis.', 'anslift')]);
    }
    update_option('aeo_auto_analyze', $enable ? 1 : 0);
    wp_send_json_success(['message' => $enable ? __('Automatic analysis enabled.', 'anslift') : __('Automatic analysis disabled.', 'anslift')]);
}
add_action('wp_ajax_aeo_toggle_auto_analyze', 'aeo_toggle_auto_analyze');
