<?php
// admin/editor-api.php

if ( ! defined( 'WPINC' ) ) { die; }

if ( ! defined( 'AEO_FAQ_ANSWER_API_URL' ) ) {
    define(
        'AEO_FAQ_ANSWER_API_URL',
        get_option( 'aeo_faq_answer_api_url', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/generate-faq-answer' )
    );
}

/**
 * Normalize FAQs to a safe, bounded array of { question, answer } strings.
 *
 * Note: The default maximum number of FAQ items has changed from 10 to 0 (unlimited).
 * Site owners can still cap the number of items via the 'aeo_editor_faq_max_items' filter.
 * Setting to 0 allows unlimited FAQs unless restricted by the filter, which may impact performance or display.
 */
function aeo_editor_normalize_faqs( $faqs_raw ) {
    // Default: 0 (unlimited). Site owners can still cap via the filter.
    $max_items       = (int) apply_filters( 'aeo_editor_faq_max_items', 0 );
    $max_q_len       = (int) apply_filters( 'aeo_editor_faq_question_max_chars', 160 );
    $max_a_len       = (int) apply_filters( 'aeo_editor_faq_answer_max_chars', 600 );
    $out = array();
    if ( is_array( $faqs_raw ) ) {
        foreach ( $faqs_raw as $faq ) {
            if ( ! is_array( $faq ) ) { continue; }
            $q = isset( $faq['question'] ) ? trim( wp_unslash( $faq['question'] ) ) : '';
            $a = isset( $faq['answer'] ) ? trim( wp_unslash( $faq['answer'] ) ) : '';
            if ( $q === '' || $a === '' ) { continue; }
            if ( $max_q_len > 0 && strlen( $q ) > $max_q_len ) { $q = substr( $q, 0, $max_q_len ); }
            if ( $max_a_len > 0 && strlen( $a ) > $max_a_len ) { $a = substr( $a, 0, $max_a_len ); }
            $out[] = array(
                'question' => sanitize_text_field( $q ),
                'answer'   => sanitize_textarea_field( $a ),
            );
            if ( $max_items > 0 && count( $out ) >= $max_items ) { break; }
        }
    }
    return $out;
}

/**
 * AJAX: Read content for editor modal
 * Params: post_id, nonce (aeo_editor_nonce)
 * Returns: { analyzed, analyzed_at, auto_status, overview, faqs[], license_active }
 */
function aeo_editor_read_ajax() {
    check_ajax_referer( 'aeo_editor_nonce', 'nonce' );

    $post_id = isset( $_POST['post_id'] ) ? (int) $_POST['post_id'] : 0;
    if ( $post_id < 1 ) {
    wp_send_json_error( array( 'code' => 'missing_post', 'message' => __( 'Missing Post ID.', 'anslift' ) ) );
    }
    if ( ! current_user_can( 'edit_post', $post_id ) ) {
    wp_send_json_error( array( 'code' => 'forbidden', 'message' => __( 'You do not have permission to edit this post.', 'anslift' ) ) );
    }
    if ( 'post' !== get_post_type( $post_id ) ) {
    wp_send_json_error( array( 'code' => 'invalid_type', 'message' => __( 'Unsupported post type.', 'anslift' ) ) );
    }

    $overview        = (string) get_post_meta( $post_id, '_aeo_ai_overview', true );
    $faqs_stored     = get_post_meta( $post_id, '_aeo_faqs', true );
    $faqs            = array();
    if ( is_array( $faqs_stored ) ) {
        foreach ( $faqs_stored as $faq ) {
            $faqs[] = array(
                'question' => isset( $faq['question'] ) ? (string) $faq['question'] : '',
                'answer'   => isset( $faq['answer'] ) ? (string) $faq['answer'] : '',
            );
        }
    }
    $analyzed        = (bool) get_post_meta( $post_id, '_aeo_plugin_analyzed', true );
    $analyzed_at     = (string) get_post_meta( $post_id, '_aeo_analyzed_at', true );
    $auto_status     = (string) get_post_meta( $post_id, '_aeo_auto_status', true );
    $license_active  = ( 'active' === get_option( 'aeo_license_status' ) );
    $trial_payload   = function_exists( 'aeo_get_trial_payload' ) ? aeo_get_trial_payload() : array();
    $improvements    = function_exists( 'aeo_impr_get' ) ? aeo_impr_get( $post_id ) : array();
    $include_debug   = function_exists( 'aeo_editor_debug_ui_enabled' ) && aeo_editor_debug_ui_enabled();

    $payload = array(
        'analyzed'       => (bool) $analyzed,
        'analyzed_at'    => (string) $analyzed_at,
        'auto_status'    => (string) $auto_status,
        'overview'       => (string) $overview,
        'faqs'           => $faqs,
        'license_active' => (bool) $license_active,
        'trial'          => $trial_payload,
        'improvements'   => is_array( $improvements ) ? $improvements : array(),
    );
    if ( $include_debug ) {
        $payload['debug_log'] = get_post_meta( $post_id, '_aeo_debug_log', true );
    }

    wp_send_json_success( $payload );
}
add_action( 'wp_ajax_aeo_editor_read', 'aeo_editor_read_ajax' );

/**
 * AJAX: Save content from editor modal (Pro only)
 * Params: post_id, overview, faqs[], nonce (aeo_editor_nonce)
 * Returns sanitized echo + analyzed_at timestamp and same shape as read
 */
function aeo_editor_save_ajax() {
    check_ajax_referer( 'aeo_editor_nonce', 'nonce' );

    $post_id = isset( $_POST['post_id'] ) ? (int) $_POST['post_id'] : 0;
    if ( $post_id < 1 ) { wp_send_json_error( array( 'code' => 'missing_post', 'message' => __( 'Missing Post ID.', 'anslift' ) ) ); }
    if ( ! current_user_can( 'edit_post', $post_id ) ) { wp_send_json_error( array( 'code' => 'forbidden', 'message' => __( 'You do not have permission to edit this post.', 'anslift' ) ) ); }
    if ( 'post' !== get_post_type( $post_id ) ) { wp_send_json_error( array( 'code' => 'invalid_type', 'message' => __( 'Unsupported post type.', 'anslift' ) ) ); }

    // License gating: Pro only for saving
    $license_active = ( 'active' === get_option( 'aeo_license_status' ) );
    if ( ! $license_active ) {
    wp_send_json_error( array( 'code' => 'license_inactive', 'message' => __( 'Saving is available to licensed users only.', 'anslift' ) ) );
    }

    $max_overview = (int) apply_filters( 'aeo_editor_overview_max_chars', 1200 );
    // Sanitize and unslash the overview input immediately to satisfy security sniffs.
    $overview     = isset( $_POST['overview'] ) ? sanitize_textarea_field( wp_unslash( $_POST['overview'] ) ) : '';
    $overview     = trim( $overview );
    if ( $max_overview > 0 && strlen( $overview ) > $max_overview ) { $overview = substr( $overview, 0, $max_overview ); }

    // Unslash and pre-sanitize FAQ payload deeply before normalization to satisfy security sniffs.
    // Use filter_input to avoid direct $_POST access
    $faqs_raw = filter_input( INPUT_POST, 'faqs', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
    $faqs_raw = is_array( $faqs_raw ) ? map_deep( $faqs_raw, 'sanitize_textarea_field' ) : array();
    $faqs     = aeo_editor_normalize_faqs( $faqs_raw );

    // Persist
    if ( $overview !== '' ) {
        update_post_meta( $post_id, '_aeo_ai_overview', $overview );
    } else {
        delete_post_meta( $post_id, '_aeo_ai_overview' );
    }
    if ( ! empty( $faqs ) ) {
        update_post_meta( $post_id, '_aeo_faqs', $faqs );
    } else {
        delete_post_meta( $post_id, '_aeo_faqs' );
    }

    // Note: Do NOT set _aeo_plugin_analyzed here (reserved for analyzer). Update analyzed_at for audit.
    $analyzed_at = current_time( 'mysql' );
    update_post_meta( $post_id, '_aeo_analyzed_at', $analyzed_at );

    // Mark that manual edits were made to prevent background auto-analysis from overwriting them.
    update_post_meta( $post_id, '_aeo_manual_edits', $analyzed_at );
    // Also clear any queued/running indicators in UI.
    update_post_meta( $post_id, '_aeo_auto_status', 'done' );

    // Compose response with current states
    $auto_status = (string) get_post_meta( $post_id, '_aeo_auto_status', true );
    $trial_payload = function_exists( 'aeo_get_trial_payload' ) ? aeo_get_trial_payload() : array();

    wp_send_json_success( array(
        'analyzed'       => (bool) get_post_meta( $post_id, '_aeo_plugin_analyzed', true ),
        'analyzed_at'    => (string) $analyzed_at,
        'auto_status'    => (string) $auto_status,
        'overview'       => (string) $overview,
        'faqs'           => $faqs,
        'license_active' => (bool) $license_active,
        'trial'          => $trial_payload,
    ) );
}
add_action( 'wp_ajax_aeo_editor_save', 'aeo_editor_save_ajax' );

/**
 * AJAX: Generate a FAQ answer (Pro only)
 * Params: post_id, question, nonce (aeo_editor_nonce)
 * Returns: { answer }
 */
function aeo_editor_generate_faq_answer_ajax() {
    check_ajax_referer( 'aeo_editor_nonce', 'nonce' );

    $post_id = filter_input( INPUT_POST, 'post_id', FILTER_VALIDATE_INT );
    if ( ! $post_id || $post_id < 1 ) {
        wp_send_json_error( array( 'code' => 'missing_post', 'message' => __( 'Missing Post ID.', 'anslift' ) ) );
    }
    if ( ! current_user_can( 'edit_post', $post_id ) ) {
        wp_send_json_error( array( 'code' => 'forbidden', 'message' => __( 'You do not have permission to edit this post.', 'anslift' ) ) );
    }
    if ( 'post' !== get_post_type( $post_id ) ) {
        wp_send_json_error( array( 'code' => 'invalid_type', 'message' => __( 'Unsupported post type.', 'anslift' ) ) );
    }

    $license_active = ( 'active' === get_option( 'aeo_license_status' ) );
    $license_key    = get_option( 'aeo_license_key' );
    if ( ! $license_active || empty( $license_key ) ) {
        wp_send_json_error( array(
            'code' => 'license_inactive',
            'message' => __( 'Generating answers is available to licensed users only.', 'anslift' ),
        ) );
    }

    $question = filter_input( INPUT_POST, 'question', FILTER_UNSAFE_RAW );
    $question = is_string( $question ) ? trim( sanitize_text_field( wp_unslash( $question ) ) ) : '';
    if ( '' === $question ) {
        wp_send_json_error( array( 'code' => 'missing_question', 'message' => __( 'Missing question.', 'anslift' ) ) );
    }

    $max_q_len = (int) apply_filters( 'aeo_faq_answer_question_max_chars', 0 );
    if ( $max_q_len > 0 && strlen( $question ) > $max_q_len ) {
        $question = substr( $question, 0, $max_q_len );
    }

    $api_url = AEO_FAQ_ANSWER_API_URL;
    $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I';
    $body = array(
        'site_url'    => home_url(),
        'license_key' => $license_key,
        'question'    => $question,
    );
    $timeout = (int) apply_filters( 'aeo_faq_answer_timeout', 20 );
    $args = array(
        'method'  => 'POST',
        'headers' => array(
            'Authorization' => 'Bearer ' . $supabase_anon_key,
            'apikey'        => $supabase_anon_key,
            'Content-Type'  => 'application/json',
        ),
        'body'    => wp_json_encode( $body ),
        'timeout' => $timeout,
    );

    $response = wp_remote_post( $api_url, $args );
    if ( is_wp_error( $response ) ) {
        wp_send_json_error( array(
            'code' => 'connection_failed',
            'message' => $response->get_error_message(),
        ) );
    }

    $code = wp_remote_retrieve_response_code( $response );
    $body_raw = wp_remote_retrieve_body( $response );
    $data = json_decode( $body_raw, true );
    if ( 200 !== (int) $code || ! is_array( $data ) ) {
        $msg = isset( $data['message'] ) ? $data['message'] : __( 'Unable to generate answer.', 'anslift' );
        wp_send_json_error( array( 'code' => 'api_error', 'message' => $msg ) );
    }

    $answer = isset( $data['answer'] ) ? sanitize_textarea_field( $data['answer'] ) : '';
    if ( '' === $answer ) {
        wp_send_json_error( array( 'code' => 'empty_answer', 'message' => __( 'Unable to generate answer.', 'anslift' ) ) );
    }

    wp_send_json_success( array( 'answer' => $answer ) );
}
add_action( 'wp_ajax_aeo_generate_faq_answer', 'aeo_editor_generate_faq_answer_ajax' );
