<?php
// admin/cleanup.php

if (!defined('WPINC')) { die; }

/**
 * Pattern fragments historically injected:
 *  - <p><strong>Overview:</strong> {text}</p>
 *  - <hr><h3>Frequently Asked Questions</h3> ... (series of <h4>Q</h4><p>A</p>)
 * We remove the leading overview paragraph if present AND the trailing FAQ block that begins with <hr><h3>Frequently Asked Questions</h3>
 */
function aeo_cleanup_legacy_injected_content_in_post($content) {
    // Remove Overview paragraph at very start
    $modified = preg_replace('/^<p><strong>Overview:<\/strong>\s*?(.*?)<\/p>/is', '', $content, 1, $count1);
    // Remove FAQ block (from <hr> then <h3>Frequently Asked Questions</h3> onwards) ONLY if it clearly matches the structure we added.
    $modified = preg_replace('/<hr>\s*<h3>Frequently Asked Questions<\/h3>.*$/is', '', $modified, 1, $count2);
    return [$modified, ($count1 + $count2) > 0];
}

/**
 * Run cleanup across all posts.
 * @return array stats
 */
function aeo_run_legacy_content_cleanup() {
    $args = [
        'post_type'      => 'post',
        'post_status'    => 'any',
        'posts_per_page' => -1,
        'fields'         => 'ids',
    ];
    $posts = get_posts($args);
    $updated = 0; $scanned = 0;
    foreach ($posts as $post_id) {
        $scanned++;
        $original = get_post_field('post_content', $post_id);
        list($cleaned, $changed) = aeo_cleanup_legacy_injected_content_in_post($original);
        if ($changed && $cleaned !== $original) {
            wp_update_post([
                'ID'           => $post_id,
                'post_content' => $cleaned,
            ]);
            $updated++;
        }
    }
    update_option('aeo_cleanup_legacy_timestamp', time());
    delete_transient('aeo_cleanup_legacy_detected');
    return ['scanned' => $scanned, 'updated' => $updated];
}

// AJAX handler (admin only)
function aeo_ajax_cleanup_legacy_content() {
    if (!current_user_can('manage_options')) { wp_send_json_error(['message' => 'Permission denied']); }
    check_ajax_referer('aeo_cleanup_nonce', 'nonce');
    $stats = aeo_run_legacy_content_cleanup();
    wp_send_json_success(['message' => 'Cleanup complete', 'stats' => $stats]);
}
add_action('wp_ajax_aeo_cleanup_legacy', 'aeo_ajax_cleanup_legacy_content');

// (Optional) WP-CLI command could be added here when developing in an environment with WP_CLI loaded.
