(function (wp) {
    const { registerPlugin } = wp.plugins;
    const { PluginDocumentSettingPanel } = wp.editPost;
    const { useSelect } = wp.data;
    const { createElement, useState, useEffect } = wp.element;
    const { Icon } = wp.components;

    const checkIcon = createElement('svg', { width: 20, height: 20, viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: 2, strokeLinecap: 'round', strokeLinejoin: 'round' },
        createElement('polyline', { points: '20 6 9 17 4 12' })
    );

    const crossIcon = createElement('svg', { width: 20, height: 20, viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: 2, strokeLinecap: 'round', strokeLinejoin: 'round' },
        createElement('line', { x1: 18, y1: 6, x2: 6, y2: 18 }),
        createElement('line', { x1: 6, y1: 6, x2: 18, y2: 18 })
    );

    const SeoChecklist = () => {
        // Get post content and type
        const { content, postType } = useSelect((select) => {
            return {
                content: select('core/editor').getEditedPostContent(),
                postType: select('core/editor').getCurrentPostType(),
            };
        }, []);

        const [hasInternalLink, setHasInternalLink] = useState(false);
        const [hasExternalLink, setHasExternalLink] = useState(false);
        const [hasImageAlt, setHasImageAlt] = useState(false);

        useEffect(() => {
            if (!content) {
                setHasInternalLink(false);
                setHasExternalLink(false);
                setHasImageAlt(false);
                return;
            }

            // Create a temporary DOM element to parse the content
            const parser = new DOMParser();
            const doc = parser.parseFromString(content, 'text/html');
            const links = doc.querySelectorAll('a');
            const images = doc.querySelectorAll('img');
            const siteUrl = window.aeoSeoData.siteUrl.replace(/\/$/, ''); // Remove trailing slash

            let internal = false;
            let external = false;

            links.forEach((link) => {
                const href = link.getAttribute('href');
                if (!href) return;

                // Check if it's an internal link (relative or starts with site URL)
                if (href.startsWith('/') || href.startsWith(siteUrl) || href.startsWith('#')) {
                    internal = true;
                } else {
                    // It's external if it starts with http/https and doesn't match site URL
                    if (href.startsWith('http') && !href.startsWith(siteUrl)) {
                        external = true;
                    }
                }
            });

            // Check if all images have alt attributes
            let allImagesHaveAlt = false;
            if (images.length > 0) {
                allImagesHaveAlt = true;
                images.forEach((img) => {
                    const alt = img.getAttribute('alt');
                    if (!alt || alt.trim() === '') {
                        allImagesHaveAlt = false;
                    }
                });
            }

            setHasInternalLink(internal);
            setHasExternalLink(external);
            setHasImageAlt(allImagesHaveAlt);

        }, [content]);

        // Only show on 'post' post type
        if (postType !== 'post') {
            return null;
        }

        return createElement(PluginDocumentSettingPanel, {
            name: 'aeo-seo-checklist',
            title: 'On-Page SEO Checklist',
            className: 'aeo-seo-panel',
            icon: 'chart-bar',
        },
            createElement('div', { className: 'aeo-checklist-container' },
                createElement('div', { className: `aeo-checklist-item ${hasInternalLink ? 'is-valid' : 'is-invalid'}` },
                    createElement('div', { className: 'aeo-checklist-icon' },
                        hasInternalLink ? checkIcon : crossIcon
                    ),
                    createElement('span', { className: 'aeo-checklist-label' }, 'Internal Links')
                ),
                createElement('div', { className: `aeo-checklist-item ${hasExternalLink ? 'is-valid' : 'is-invalid'}` },
                    createElement('div', { className: 'aeo-checklist-icon' },
                        hasExternalLink ? checkIcon : crossIcon
                    ),
                    createElement('span', { className: 'aeo-checklist-label' }, 'External Links')
                ),
                createElement('div', { className: `aeo-checklist-item ${hasImageAlt ? 'is-valid' : 'is-invalid'}` },
                    createElement('div', { className: 'aeo-checklist-icon' },
                        hasImageAlt ? checkIcon : crossIcon
                    ),
                    createElement('span', { className: 'aeo-checklist-label' }, 'Image Alt Text')
                )
            )
        );
    };

    registerPlugin('aeo-seo-sidebar', {
        render: SeoChecklist,
        icon: 'chart-bar',
    });

})(window.wp);
