const HEADER_MOUNT_MAX_RETRIES = 100;
const LONG_SEGMENT_CHAR_LIMIT = 300;

(function () {
    if (!window.aeo_editor_object) { return; }
    const cfg = window.aeo_editor_object;
    // Only attach on valid post editor for post type 'post'
    if (!cfg || (!cfg.post_id && cfg.post_id !== 0)) { return; }

    function normalizeTrialData(trial) {
        const parsed = {
            limit: 0,
            used: 0,
            remaining: 0,
            notice: ''
        };
        if (trial && typeof trial === 'object') {
            if (typeof trial.limit !== 'undefined') {
                const lim = parseInt(trial.limit, 10);
                if (!isNaN(lim)) { parsed.limit = lim; }
            }
            if (typeof trial.used !== 'undefined') {
                const used = parseInt(trial.used, 10);
                if (!isNaN(used)) { parsed.used = Math.max(0, used); }
            }
            if (typeof trial.remaining !== 'undefined') {
                const remain = parseInt(trial.remaining, 10);
                if (!isNaN(remain)) { parsed.remaining = Math.max(0, remain); }
            }
            if (typeof trial.notice === 'string') {
                parsed.notice = trial.notice;
            }
        }
        if (!Number.isFinite(parsed.remaining) || parsed.remaining === 0 && parsed.used < parsed.limit) {
            parsed.remaining = Math.max(0, parsed.limit - parsed.used);
        }
        return parsed;
    }

    function hasClassicEditorDom() {
        return !!(
            document.getElementById('wp-content-wrap') ||
            document.getElementById('postdivrich') ||
            document.getElementById('content_ifr')
        );
    }

    function detectClassicEditorMode() {
        const wrap = document.getElementById('wp-content-wrap');
        if (wrap) {
            if (wrap.classList.contains('html-active')) {
                return 'classic-text';
            }
            if (wrap.classList.contains('tmce-active')) {
                return 'classic-visual';
            }
        }
        if (window.tinymce && typeof window.tinymce.activeEditor !== 'undefined' && window.tinymce.activeEditor) {
            return 'classic-visual';
        }
        return 'classic-text';
    }

    function detectEditorMode(cfgObj) {
        if (hasClassicEditorDom()) {
            return detectClassicEditorMode();
        }
        if (cfgObj && cfgObj.is_gutenberg) {
            return 'gutenberg';
        }
        return detectClassicEditorMode();
    }

    const initialEditorMode = detectEditorMode(cfg);
    const featureFlags = Object.assign({ classic_overlays_enabled: true, debug_ui: false }, cfg.flags || {});

    // Expose env
    window.aeoEditor = window.aeoEditor || {};
    window.aeoEditor.bootstrapped = true;
    window.aeoEditor.env = {
        postId: cfg.post_id || 0,
        editorMode: initialEditorMode,
        isGutenberg: initialEditorMode === 'gutenberg',
        licenseActive: !!cfg.license_active,
        analyzed: !!cfg.current_post_analyzed,
        bulkActive: !!featureFlags.bulk_active,
        autoStatus: '',
        nonces: cfg.nonces || {},
        flags: featureFlags
    };
    window.aeoEditor.env.classicOverlaysEnabled = (typeof featureFlags.classic_overlays_enabled === 'undefined') ? true : !!featureFlags.classic_overlays_enabled;
    window.aeoEditor.env.trial = normalizeTrialData(cfg.trial);
    const debugUiEnabled = !!featureFlags.debug_ui;
    const debugSegments = !!featureFlags.debug_segments;

    function setEditorMode(nextMode) {
        if (!nextMode || typeof nextMode !== 'string') { return; }
        const normalized = nextMode;
        if (window.aeoEditor.env.editorMode === normalized) { return; }
        window.aeoEditor.env.editorMode = normalized;
        window.aeoEditor.env.isGutenberg = normalized === 'gutenberg';
        try { updateButtonVisibility(); } catch (_) { }
        try { mountButtonForEditor(); } catch (_) { }
        try { ensureCountBadge(); } catch (_) { }
    }

    function getTrialState() {
        return window.aeoEditor.env.trial || normalizeTrialData();
    }

    function trialRunsRemaining() {
        const trial = getTrialState();
        const remaining = parseInt(trial.remaining, 10);
        return isNaN(remaining) ? 0 : Math.max(0, remaining);
    }

    function trialRunsUsed() {
        const trial = getTrialState();
        const used = parseInt(trial.used, 10);
        return isNaN(used) ? 0 : Math.max(0, used);
    }

    function trialLimit() {
        const trial = getTrialState();
        const limit = parseInt(trial.limit, 10);
        return isNaN(limit) ? 0 : Math.max(0, limit);
    }

    function hasTrialAccess() {
        return !!window.aeoEditor.env.licenseActive || trialRunsRemaining() > 0;
    }

    function getAnalyzeSaveRequiredMessage() {
        return (window.wp && wp.i18n)
            ? wp.i18n.__('Save the Post before analyzing', 'aeo-content-plugin')
            : 'Save the Post before analyzing';
    }

    function normalizePostAttributeValue(value) {
        if (typeof value === 'string') { return value; }
        if (value && typeof value === 'object') {
            if (typeof value.raw === 'string') { return value.raw; }
            if (typeof value.rendered === 'string') { return value.rendered; }
        }
        return '';
    }

    let classicBaseline = null;
    function getClassicTitleValue() {
        const input = document.getElementById('title');
        return input && typeof input.value === 'string' ? input.value : '';
    }

    function getClassicSavedTitleValue() {
        const original = document.getElementById('original_post_title');
        if (original && typeof original.value === 'string') { return original.value; }
        const base = ensureClassicBaseline();
        return base ? base.title : '';
    }

    function getClassicContentValue() {
        try {
            const editor = getClassicTinyMCEEditor();
            if (editor && typeof editor.getContent === 'function' && (!editor.isHidden || !editor.isHidden())) {
                return editor.getContent({ format: 'raw' }) || editor.getContent() || '';
            }
        } catch (_) { /* noop */ }
        const textarea = getClassicTextarea();
        return textarea && typeof textarea.value === 'string' ? textarea.value : '';
    }

    function ensureClassicBaseline() {
        if (classicBaseline) { return classicBaseline; }
        classicBaseline = {
            title: getClassicTitleValue(),
            content: getClassicContentValue()
        };
        return classicBaseline;
    }

    function getClassicPostStatus() {
        const original = document.getElementById('original_post_status');
        if (original && typeof original.value === 'string') { return original.value; }
        const status = document.getElementById('post_status');
        if (status && typeof status.value === 'string') { return status.value; }
        return '';
    }

    function isClassicPostSavedForAnalyze() {
        const msg = getAnalyzeSaveRequiredMessage();
        const status = getClassicPostStatus();
        const isAutoDraft = status === 'auto-draft' || status === 'new';
        if (isAutoDraft) {
            return { saved: false, reason: msg };
        }
        const title = getClassicTitleValue();
        const savedTitle = getClassicSavedTitleValue();
        const titleDirty = title !== savedTitle;

        let contentDirty = false;
        const editor = getClassicTinyMCEEditor();
        if (editor && typeof editor.isDirty === 'function' && (!editor.isHidden || !editor.isHidden())) {
            contentDirty = !!editor.isDirty();
            if (!contentDirty && classicBaseline) {
                classicBaseline.content = getClassicContentValue();
            }
        } else {
            const base = ensureClassicBaseline();
            const content = getClassicContentValue();
            contentDirty = (content !== base.content);
        }

        return (titleDirty || contentDirty) ? { saved: false, reason: msg } : { saved: true, reason: '' };
    }

    function isGutenbergPostSavedForAnalyze() {
        const msg = getAnalyzeSaveRequiredMessage();
        try {
            const sel = (window.wp && wp.data && wp.data.select) ? wp.data.select('core/editor') : null;
            if (!sel) { return { saved: true, reason: '' }; }
            let status = '';
            if (typeof sel.getEditedPostAttribute === 'function') {
                status = sel.getEditedPostAttribute('status') || '';
            }
            const isAutoDraft = status === 'auto-draft' || status === 'new';
            let isDirty = false;
            if (typeof sel.isEditedPostDirty === 'function') {
                isDirty = !!sel.isEditedPostDirty();
            } else if (typeof sel.isClean === 'function') {
                isDirty = !sel.isClean();
            } else {
                try {
                    const editedTitle = normalizePostAttributeValue(sel.getEditedPostAttribute ? sel.getEditedPostAttribute('title') : '');
                    const savedTitle = normalizePostAttributeValue(sel.getCurrentPostAttribute ? sel.getCurrentPostAttribute('title') : '');
                    const editedContent = normalizePostAttributeValue(sel.getEditedPostAttribute ? sel.getEditedPostAttribute('content') : '');
                    const savedContent = normalizePostAttributeValue(sel.getCurrentPostAttribute ? sel.getCurrentPostAttribute('content') : '');
                    if (editedTitle !== '' || savedTitle !== '') {
                        if (editedTitle !== savedTitle) { isDirty = true; }
                    }
                    if (editedContent !== '' || savedContent !== '') {
                        if (editedContent !== savedContent) { isDirty = true; }
                    }
                } catch (_) { /* noop */ }
            }
            const saved = !isAutoDraft && !isDirty;
            return saved ? { saved: true, reason: '' } : { saved: false, reason: msg };
        } catch (_) {
            return { saved: true, reason: '' };
        }
    }

    function getAnalyzeSaveGate() {
        if (isGutenbergEnv()) {
            return isGutenbergPostSavedForAnalyze();
        }
        return isClassicPostSavedForAnalyze();
    }

    function getGutenbergSaveGateKey() {
        try {
            const sel = (window.wp && wp.data && wp.data.select) ? wp.data.select('core/editor') : null;
            if (!sel) { return ''; }
            let status = '';
            if (typeof sel.getEditedPostAttribute === 'function') {
                status = sel.getEditedPostAttribute('status') || '';
            }
            let dirty = false;
            if (typeof sel.isEditedPostDirty === 'function') {
                dirty = !!sel.isEditedPostDirty();
            } else if (typeof sel.isClean === 'function') {
                dirty = !sel.isClean();
            }
            return status + '|' + (dirty ? 'dirty' : 'clean');
        } catch (_) {
            return '';
        }
    }

    function setTrialState(next) {
        const merged = normalizeTrialData(Object.assign({}, getTrialState(), next || {}));
        window.aeoEditor.env.trial = merged;
        updateAnalyzeButton();
        updateLicenseBanner();
    }

    function areClassicOverlaysEnabled() {
        const flags = (window.aeoEditor && window.aeoEditor.env && window.aeoEditor.env.flags) ? window.aeoEditor.env.flags : null;
        if (!flags || typeof flags.classic_overlays_enabled === 'undefined') {
            return true;
        }
        return !!flags.classic_overlays_enabled;
    }

    // Phase 5: Fetch server-truth usage status for gating (Free vs Pro)
    function refreshUsageStatus() {
        try {
            const url = (cfg && cfg.ajax_url) ? cfg.ajax_url : null;
            const nonces = (window.aeoEditor && window.aeoEditor.env && window.aeoEditor.env.nonces) ? window.aeoEditor.env.nonces : {};
            const nonce = nonces && nonces.status ? nonces.status : '';
            if (!url || !nonce) return;
            const params = new URLSearchParams();
            params.append('action', 'aeo_usage_status');
            params.append('nonce', nonce);
            fetch(url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
                body: params.toString(),
                credentials: 'same-origin'
            })
                .then(function (r) { return r.json(); })
                .then(function (resp) {
                    if (!resp || !resp.success || !resp.data) { return; }
                    applyUsageStatusToEditor(resp.data);
                })
                .catch(function () { /* silent */ });
        } catch (_) { /* noop */ }
    }

    function applyUsageStatusToEditor(status) {
        if (!status || typeof status !== 'object') return;
        const plan = String(status.plan || '').toLowerCase();
        const isPro = (plan === 'pro');
        window.aeoEditor.env.licenseActive = !!isPro;
        if (isPro) {
            // Pro: no trial gating; backend enforces 429 if quota exceeded
            setTrialState({ limit: 0, used: 0, remaining: 0, notice: '' });
        } else {
            const free = status.free || {};
            const limit = parseInt(free.limit || 10, 10) || 10;
            const used = parseInt(free.used || 0, 10) || 0;
            const remaining = Math.max(0, (parseInt(free.remaining, 10) || (limit - used)));
            let notice = '';
            try {
                if (window.wp && wp.i18n && typeof wp.i18n.sprintf === 'function') {
                    notice = remaining > 0
                        ? wp.i18n.sprintf(wp.i18n.__('Free preview: %1$d of %2$d runs remaining.', 'aeo-content-plugin'), remaining, limit)
                        : wp.i18n.__('Free preview exhausted. Activate your license to continue.', 'aeo-content-plugin');
                } else {
                    notice = remaining > 0
                        ? ('Free preview: ' + remaining + ' of ' + limit + ' runs remaining.')
                        : 'Free preview exhausted. Activate your license to continue.';
                }
            } catch (_) { /* noop */ }
            setTrialState({ limit: limit, used: used, remaining: remaining, notice: notice });
        }
    }

    const hasPostId = !!(window.aeoEditor.env.postId && window.aeoEditor.env.postId > 0);

    const debugTabButton = debugUiEnabled
        ? '<button role="tab" aria-selected="false" aria-controls="aeo-tab-debug" id="aeo-tab-btn-debug" class="aeo-tab">' + (window.wp && wp.i18n ? wp.i18n.__('Debug', 'aeo-content-plugin') : 'Debug') + '</button>'
        : '';
    const debugTabSection = debugUiEnabled
        ? [
            '<section id="aeo-tab-debug" class="aeo-section" role="tabpanel" aria-labelledby="aeo-tab-btn-debug">',
            '<div class="aeo-debug-head">',
            '<strong>' + (window.wp && wp.i18n ? wp.i18n.__('Debug Log', 'aeo-content-plugin') : 'Debug Log') + '</strong>',
            '<button type="button" class="button" id="aeo-debug-copy">' + (window.wp && wp.i18n ? wp.i18n.__('Copy Log', 'aeo-content-plugin') : 'Copy Log') + '</button>',
            '</div>',
            '<textarea id="aeo-debug-log" class="aeo-textarea aeo-debug-log" readonly></textarea>',
            '</section>'
        ].join('')
        : '';

    // Create FAB + Modal container once
    const rootId = 'aeo-editor-root';
    if (document.getElementById(rootId)) { return; }
    const root = document.createElement('div');
    root.id = rootId;
    root.className = 'aeo-root';
    root.innerHTML = [
        // Button container; will move into header on Gutenberg
        '<div class="aeo-fab' + (hasPostId ? ' aeo-visible' : '') + '" id="aeo-button-container"><button type="button" class="button button-primary aeo-open-editor-btn" id="aeo-open-editor"><span class="aeo-open-label">' + (cfg.labels && cfg.labels.open_modal ? cfg.labels.open_modal : 'Open AEO Editor') + '</span><span class="aeo-count-badge is-empty" id="aeo-count-badge" aria-label="' + (window.wp && wp.i18n ? wp.i18n.__('Pending suggestions', 'aeo-content-plugin') : 'Pending suggestions') + '"></span></button></div>',
        // Modal shell (placeholder)
        '<div class="aeo-modal" id="aeo-modal" aria-hidden="true" role="dialog" aria-modal="true" aria-label="AEO Editor">',
        '<div class="aeo-backdrop" data-aeo-close></div>',
        '<div class="aeo-dialog" tabindex="-1">',
        '<div class="aeo-modal-header">',
        '<div class="aeo-modal-title-row">',
        '<strong class="aeo-modal-title">AEO Editor</strong>',
        '<div class="aeo-modal-title-actions">',
        // Close icon button (replaces text button). Uses an accessible label + title.
        '<button type="button" class="aeo-icon-btn" id="aeo-modal-close" aria-label="' + (window.wp && wp.i18n ? wp.i18n.__('Close modal', 'aeo-content-plugin') : 'Close modal') + '" title="' + (window.wp && wp.i18n ? wp.i18n.__('Close', 'aeo-content-plugin') : 'Close') + '">\n'
        + '<svg class="aeo-icon" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">'
        + '<line x1="18" y1="6" x2="6" y2="18"></line>'
        + '<line x1="6" y1="6" x2="18" y2="18"></line>'
        + '</svg>'
        + '</button>',
        '</div>',
        '</div>',
        '<div class="aeo-tabs" role="tablist">',
        '<button role="tab" aria-selected="true" aria-controls="aeo-tab-overview" id="aeo-tab-btn-overview" class="aeo-tab is-active">' + (window.wp && wp.i18n ? wp.i18n.__('Summary', 'aeo-content-plugin') : 'Summary') + '</button>',
        '<button role="tab" aria-selected="false" aria-controls="aeo-tab-faqs" id="aeo-tab-btn-faqs" class="aeo-tab">' + (window.wp && wp.i18n ? wp.i18n.__('FAQs', 'aeo-content-plugin') : 'FAQs') + '</button>',
        debugTabButton,
        '</div>',
        '</div>',
        '<div class="aeo-license-banner" id="aeo-license-banner"></div>',
        '<div class="aeo-modal-body">',
        '<section id="aeo-tab-overview" class="aeo-section is-active" role="tabpanel" aria-labelledby="aeo-tab-btn-overview">',
        '<label class="aeo-label" for="aeo-overview">' + (window.wp && wp.i18n ? wp.i18n.__('AI Summary', 'aeo-content-plugin') : 'AI Summary') + '</label>',
        '<textarea id="aeo-overview" class="aeo-textarea" placeholder="Write or paste your summary…"></textarea>',
        '<div class="aeo-helper"><span id="aeo-overview-count">0</span></div>',
        '</section>',
        '<section id="aeo-tab-faqs" class="aeo-section" role="tabpanel" aria-labelledby="aeo-tab-btn-faqs">',
        '<div class="aeo-faqs-head">',
        '<strong>' + (window.wp && wp.i18n ? wp.i18n.__('Frequently Asked Questions', 'aeo-content-plugin') : 'Frequently Asked Questions') + '</strong>',
        '<button type="button" class="button button-secondary" id="aeo-add-faq">' + (window.wp && wp.i18n ? wp.i18n.__('Add FAQ', 'aeo-content-plugin') : 'Add FAQ') + '</button>',
        // Read-only overlay (shown when license inactive)
        '<div id="aeo-readonly-overlay" class="aeo-readonly-overlay"></div>',
        '</div>',
        '<div id="aeo-faqs-list" class="aeo-faqs-list"></div>',
        '<p class="aeo-helper aeo-faqs-empty" id="aeo-faqs-empty">' + (window.wp && wp.i18n ? wp.i18n.__('No FAQs yet. Use “Add FAQ” to create one.', 'aeo-content-plugin') : 'No FAQs yet. Use “Add FAQ” to create one.') + '</p>',
        '</section>',
        debugTabSection,
        '</div>',
        '<div class="aeo-modal-footer">',
        '<button type="button" class="button" id="aeo-modal-cancel">' + (window.wp && wp.i18n ? wp.i18n.__('Cancel', 'aeo-content-plugin') : 'Cancel') + '</button>',
        '<button type="button" class="button button-primary" id="aeo-modal-save" disabled>' + (window.wp && wp.i18n ? wp.i18n.__('Save', 'aeo-content-plugin') : 'Save') + '</button>',
        '<button type="button" class="button" id="aeo-modal-analyze" disabled>' + (window.wp && wp.i18n ? wp.i18n.__('Analyze', 'aeo-content-plugin') : 'Analyze') + '</button>',
        '<a href="admin.php?page=aeo-content-plugin" class="button button-primary" id="aeo-license-cta" style="display:none;">' + (window.wp && wp.i18n ? wp.i18n.__('Activate license', 'aeo-content-plugin') : 'Activate license') + '</a>',
        '</div>',
        // Loading overlay (covers entire dialog while analyzing)
        '<div class="aeo-modal-loading" id="aeo-modal-loading" aria-hidden="true">',
        '  <div class="aeo-loading-spinner" aria-hidden="true"></div>',
        '  <div class="aeo-modal-loading-text">' + (window.wp && wp.i18n ? wp.i18n.__('Analyzing…', 'aeo-content-plugin') : 'Analyzing…') + '</div>',
        '</div>',
        '</div>',
        '</div>'
    ].join('');

    // Prefer mounting late in body to avoid layout shifts
    document.body.appendChild(root);

    const fabBtn = document.getElementById('aeo-open-editor');
    const buttonContainer = document.getElementById('aeo-button-container');
    const modal = document.getElementById('aeo-modal');
    const closeBtn = document.getElementById('aeo-modal-close');
    const cancelBtn = document.getElementById('aeo-modal-cancel');
    const backdrop = modal ? modal.querySelector('[data-aeo-close]') : null;
    const tabBtnOverview = document.getElementById('aeo-tab-btn-overview');
    const tabBtnFaqs = document.getElementById('aeo-tab-btn-faqs');
    const sectionOverview = document.getElementById('aeo-tab-overview');
    const sectionFaqs = document.getElementById('aeo-tab-faqs');
    const overviewEl = document.getElementById('aeo-overview');
    const overviewCount = document.getElementById('aeo-overview-count');
    const addFaqBtn = document.getElementById('aeo-add-faq');
    const faqsList = document.getElementById('aeo-faqs-list');
    const faqsEmpty = document.getElementById('aeo-faqs-empty');
    const licenseBanner = document.getElementById('aeo-license-banner');
    const readonlyOverlay = document.getElementById('aeo-readonly-overlay');
    const saveBtn = document.getElementById('aeo-modal-save');
    const analyzeBtn = document.getElementById('aeo-modal-analyze');
    const licenseCta = document.getElementById('aeo-license-cta');
    const loadingOverlay = document.getElementById('aeo-modal-loading');
    const loadingText = loadingOverlay ? loadingOverlay.querySelector('.aeo-modal-loading-text') : null;

    // Debug UI elements
    const tabBtnDebug = document.getElementById('aeo-tab-btn-debug');
    const sectionDebug = document.getElementById('aeo-tab-debug');
    const debugLogEl = document.getElementById('aeo-debug-log');
    const debugCopyBtn = document.getElementById('aeo-debug-copy');

    let classicTextHint = null;

    function ensureClassicTextHint() {
        if (!classicTextHint) {
            classicTextHint = document.createElement('p');
            classicTextHint.id = 'aeo-classic-text-hint';
            classicTextHint.className = 'notice notice-info aeo-classic-text-hint';
            classicTextHint.style.display = 'none';
            classicTextHint.style.marginTop = '8px';
            classicTextHint.style.padding = '8px 12px';
            classicTextHint.style.fontSize = '13px';
            const msg = (window.wp && wp.i18n
                ? wp.i18n.__('Switch back to the Visual tab to view inline suggestions.', 'aeo-content-plugin')
                : 'Switch back to the Visual tab to view inline suggestions.');
            classicTextHint.textContent = msg;
            root.appendChild(classicTextHint);
        }
        const classicWrap = document.getElementById('aeo-classic-btn-wrap');
        if (classicWrap && classicWrap.parentNode && classicTextHint.parentNode !== classicWrap.parentNode) {
            classicWrap.parentNode.insertBefore(classicTextHint, classicWrap.nextSibling);
        }
        return classicTextHint;
    }

    function toggleClassicTextHint(visible) {
        const el = ensureClassicTextHint();
        if (!el) return;
        el.style.display = visible ? 'block' : 'none';
    }

    // Phase 6/9 state
    let initialState = { overview: '', faqs: [] };
    let isDirty = false;
    let isLoading = false;
    let isAnalyzing = false;
    let isAnalysisPolling = false;
    let analysisPollTimer = null;
    let analysisPollStartedAt = 0;
    let analysisPollInFlight = false;

    let lastFocus = null;
    function showOverlay(message) {
        if (!loadingOverlay) return;
        if (typeof message === 'string' && loadingText) {
            loadingText.textContent = message;
        }
        loadingOverlay.classList.add('is-visible');
        loadingOverlay.setAttribute('aria-hidden', 'false');
    }
    function hideOverlay() {
        if (!loadingOverlay) return;
        loadingOverlay.classList.remove('is-visible');
        loadingOverlay.setAttribute('aria-hidden', 'true');
    }
    function openModal() {
        if (!modal) return;
        lastFocus = document.activeElement;
        modal.classList.add('aeo-open');
        document.body.classList.add('aeo-modal-open');
        // Set focus inside dialog
        const dlg = modal.querySelector('.aeo-dialog');
        if (dlg) { dlg.focus(); }
        modal.setAttribute('aria-hidden', 'false');
        // Load the latest data when opened
        try { console.debug && console.debug('[AEO] Modal open; licenseActive=', window.aeoEditor.env.licenseActive, 'postId=', window.aeoEditor.env.postId); } catch (_) { }
        showOverlay((window.wp && wp.i18n ? wp.i18n.__('Loading…', 'aeo-content-plugin') : 'Loading…'));
        // Align gating with latest server usage on open
        try { refreshUsageStatus(); } catch (_) { }
        loadData();
        // Also refresh improvements in the background and decorate once loaded
        try {
            fetchImprovements().then(function () { try { decorateEditorSurface(); } catch (_) { } });
        } catch (_) { /* noop */ }
    }
    function closeModal() {
        if (!modal) return;
        modal.classList.remove('aeo-open');
        document.body.classList.remove('aeo-modal-open');
        modal.setAttribute('aria-hidden', 'true');
        if (lastFocus && typeof lastFocus.focus === 'function') {
            lastFocus.focus();
        } else if (fabBtn) {
            fabBtn.focus();
        }
    }

    function getEditedPostContentHtml() {
        if (!isGutenbergEnv()) { return ''; }
        try {
            const sel = wp.data && wp.data.select ? wp.data.select('core/editor') : null;
            if (sel && typeof sel.getEditedPostContent === 'function') {
                return sel.getEditedPostContent() || '';
            }
        } catch (_) { /* noop */ }
        return '';
    }

    // Add this new function near your other helpers
    function segmentEditorContent() {
        let payload = null;
        if (isGutenbergEnv()) {
            try {
                const sel = wp.data.select('core/block-editor');
                const blocks = sel.getBlocks();
                let items = [];

                function getBlockHtmlContent(block) {
                    if (!block || !block.attributes) { return ''; }
                    if (typeof block.attributes.content === 'string') {
                        return block.attributes.content;
                    }
                    if (typeof block.attributes.value === 'string') {
                        return block.attributes.value;
                    }
                    if (typeof block.attributes.html === 'string') {
                        return block.attributes.html;
                    }
                    return '';
                }

                (function walk(blockList) {
                    (blockList || []).forEach(function (b) {
                        const name = b ? b.name : '';
                        // Support paragraphs, headings, list items, and pullquotes
                        // We skip core/quote container because it usually contains innerBlocks (paragraphs) which we recurse into.
                        const isSupported = (
                            name === 'core/paragraph' ||
                            name === 'core/heading' ||
                            name === 'core/list-item' ||
                            name === 'core/pullquote'
                        );
                        const isHtmlBlock = (
                            name === 'core/freeform' ||
                            name === 'core/html'
                        );

                        if (isHtmlBlock) {
                            const rawHtml = getBlockHtmlContent(b);
                            if (rawHtml) {
                                const segments = extractParagraphSegmentsFromHtml(rawHtml);
                                if (segments.length) {
                                    segments.forEach(function (seg) {
                                        seg.blockType = name;
                                        seg.clientId = b.clientId;
                                    });
                                    items = buildSegmentItemsFromSegments(segments, items);
                                } else {
                                    const text = normalizeTextForHash(rawHtml);
                                    if (text) {
                                        const hints = detectStructuralHints(text);
                                        const isLong = isLongSegment(text);
                                        items.push({
                                            index: items.length,
                                            text: text,
                                            hash: sha1Sync(text),
                                            blockType: name,
                                            clientId: b.clientId,
                                            html: rawHtml || '',
                                            hints: hints,
                                            is_long: isLong
                                        });
                                    }
                                }
                            }
                        } else if (isSupported) {
                            const content = getBlockHtmlContent(b);
                            const text = normalizeTextForHash(content);
                            if (text) {
                                const hints = detectStructuralHints(text);
                                const isLong = isLongSegment(text);
                                items.push({
                                    index: items.length,
                                    text: text,
                                    hash: sha1Sync(text),
                                    blockType: name,
                                    clientId: b.clientId,
                                    html: content || '',
                                    hints: hints,
                                    is_long: isLong
                                });
                            }
                        }
                        if (b.innerBlocks && b.innerBlocks.length) {
                            walk(b.innerBlocks);
                        }
                    });
                })(blocks);

                if (items.length <= 1) {
                    const fullHtml = getEditedPostContentHtml();
                    if (fullHtml && fullHtml.length > 800) {
                        const segments = extractParagraphSegmentsFromHtml(fullHtml);
                        if (segments.length > items.length) {
                            items = buildSegmentItemsFromSegments(segments);
                        }
                    }
                }

                payload = { editor: 'gutenberg', items: items };
            } catch (_) {
                payload = null;
            }
        } else {
            payload = segmentClassicContent();
        }

        if (payload && debugSegments) {
            try { console.debug('[AEO] segmentEditorContent', payload); } catch (_) { }
        }

        return payload;
    }

    function isLongSegment(text) {
        if (!text) { return false; }
        return text.length > LONG_SEGMENT_CHAR_LIMIT;
    }

    function detectStructuralHints(text) {
        const out = { steps: false, comparison: false, list: false };
        if (!text) { return out; }
        const t = String(text).toLowerCase();
        const stepMatches = (t.match(/\bstep\s*\d+\b/g) || []).length;
        const ordMatches = (t.match(/\b(first|second|third|fourth|fifth|sixth|seventh|eighth|ninth|tenth|next|then|finally|last|lastly)\b/g) || []).length;
        const numMatches = (t.match(/(?:^|\s)\d{1,2}[.)-]\s+[a-z]/g) || []).length;
        out.steps = (stepMatches + ordMatches + numMatches) >= 2;
        const comparisonCue = /\b(vs\.?|versus|compared to|compared with|compare|comparison|difference between|differences between|pros and cons|pros\/cons|advantages and disadvantages|better|worse|faster|slower|cheaper|more expensive|less expensive|higher|lower|larger|smaller|heavier|lighter|on the other hand|in contrast|whereas)\b/.test(t);
        const numberCount = (t.match(/\b\d+(?:\.\d+)?\b/g) || []).length;
        const modelTokenCount = (t.match(/\b[a-z]{1,6}\d{1,4}\b/g) || []).length;
        const compareJoiner = /\b(and|or|both|either|neither|than|while)\b/.test(t);
        out.comparison = comparisonCue || ((numberCount >= 4 || modelTokenCount >= 2) && compareJoiner);
        const listTrigger = /\b(include|includes|including|such as|for example|e\.g\.|examples)\b/.test(t) || t.indexOf(':') !== -1;
        const commaCount = (t.match(/,/g) || []).length;
        const semicolonCount = (t.match(/;/g) || []).length;
        out.list = (listTrigger && commaCount >= 2) || semicolonCount >= 2;
        return out;
    }

    function segmentClassicContent() {
        if (isGutenbergEnv()) { return null; }
        const mode = detectClassicEditorMode();
        const editor = getClassicTinyMCEEditor();
        const textarea = getClassicTextarea();
        let items = [];

        if (mode === 'classic-visual' && editor) {
            items = segmentClassicVisual(editor);
        }

        if (!items.length) {
            const source = textarea && textarea.value ? textarea.value : '';
            items = segmentClassicText(source);
        }

        return {
            editor: 'classic',
            mode: mode,
            items: items
        };
    }

    function segmentClassicVisual(editor) {
        try {
            const html = editor && typeof editor.getContent === 'function'
                ? (editor.getContent({ format: 'html' }) || editor.getContent() || '')
                : '';
            let activeHtml = html;
            if (!activeHtml && editor && editor.getBody && editor.getBody()) {
                activeHtml = editor.getBody().innerHTML || '';
            }
            if (!activeHtml && editor && editor.iframeElement && editor.iframeElement.contentDocument) {
                const doc = editor.iframeElement.contentDocument;
                activeHtml = (doc.body && doc.body.innerHTML) || '';
            }
            if (!activeHtml) {
                const iframeId = editor && editor.id ? editor.id + '_ifr' : null;
                const iframe = iframeId ? document.getElementById(iframeId) : null;
                if (iframe && iframe.contentDocument && iframe.contentDocument.body) {
                    activeHtml = iframe.contentDocument.body.innerHTML || '';
                }
            }
            const segments = extractParagraphSegmentsFromHtml(activeHtml);
            if (segments.length) {
                return buildSegmentItemsFromSegments(segments);
            }
            return buildSegmentItemsFromTexts(extractParagraphTextsFromHtml(activeHtml));
        } catch (_) {
            return [];
        }
    }

    function segmentClassicText(rawText) {
        if (!rawText) {
            return [];
        }
        const looksLikeHtml = /<[a-z][\s\S]*>/i.test(rawText);
        if (looksLikeHtml) {
            const segments = extractParagraphSegmentsFromHtml(rawText);
            if (segments.length) {
                return buildSegmentItemsFromSegments(segments);
            }
            return buildSegmentItemsFromTexts(extractParagraphTextsFromHtml(rawText));
        }
        const normalized = rawText.replace(/\r\n/g, '\n');
        const chunks = normalized.split(/\n{2,}/).map(function (part) {
            return part.replace(/\n+/g, ' ').trim();
        }).filter(function (part) { return !!part; });
        if (!chunks.length) {
            return [];
        }
        return buildSegmentItemsFromTexts(chunks);
    }

    function extractParagraphSegmentsFromHtml(html) {
        if (!html) { return []; }
        const tmp = document.createElement('div');
        tmp.innerHTML = html;
        const nodes = tmp.querySelectorAll('p, li, h1, h2, h3, h4, h5, h6');
        const segments = [];
        nodes.forEach(function (node) {
            const text = node.textContent || node.innerText || '';
            if (text && text.trim().length) {
                segments.push({
                    text: text,
                    html: node.innerHTML || ''
                });
            }
        });
        if (!segments.length) {
            const fallback = tmp.textContent || tmp.innerText || '';
            if (fallback && fallback.trim().length) {
                segments.push({ text: fallback, html: '' });
            }
        }
        return segments;
    }

    function extractParagraphTextsFromHtml(html) {
        if (!html) { return []; }
        const tmp = document.createElement('div');
        tmp.innerHTML = html;
        // Removed blockquote to avoid duplication with inner paragraphs
        const selectors = tmp.querySelectorAll('p, li, h1, h2, h3, h4, h5, h6');
        const texts = [];
        selectors.forEach(function (node) {
            const text = node.textContent || node.innerText || '';
            if (text && text.trim().length) {
                texts.push(text);
            }
        });
        if (!texts.length) {
            const fallback = tmp.textContent || tmp.innerText || '';
            if (fallback && fallback.trim().length) {
                texts.push(fallback);
            }
        }
        return texts;
    }

    function buildSegmentItemsFromSegments(segments, seedItems) {
        const items = Array.isArray(seedItems) ? seedItems : [];
        (segments || []).forEach(function (seg) {
            const rawText = (seg && typeof seg.text === 'string') ? seg.text : '';
            const text = normalizeTextForHash(rawText);
            if (!text) { return; }
            const hints = detectStructuralHints(text);
            const isLong = isLongSegment(text);
            const html = (seg && typeof seg.html === 'string') ? seg.html : '';
            const item = {
                index: items.length,
                text: text,
                hash: sha1Sync(text),
                html: html,
                hints: hints,
                is_long: isLong
            };
            if (seg && seg.blockType) { item.blockType = seg.blockType; }
            if (seg && seg.clientId) { item.clientId = seg.clientId; }
            items.push(item);
        });
        return items;
    }

    function buildSegmentItemsFromTexts(texts, seedItems) {
        const items = Array.isArray(seedItems) ? seedItems : [];
        (texts || []).forEach(function (raw) {
            const text = normalizeTextForHash(raw);
            if (!text) { return; }
            const hash = sha1Sync(text);
            const hints = detectStructuralHints(text);
            const isLong = isLongSegment(text);
            items.push({
                index: items.length,
                text: text,
                hash: hash,
                hints: hints,
                is_long: isLong
            });
        });
        return items;
    }

    function getClassicTextarea() {
        const activeId = (window.wpActiveEditor && typeof window.wpActiveEditor === 'string') ? window.wpActiveEditor : 'content';
        return document.getElementById(activeId) || document.getElementById('content');
    }

    function getClassicTinyMCEEditor() {
        if (!window.tinymce) { return null; }
        const activeId = (window.wpActiveEditor && typeof window.wpActiveEditor === 'string') ? window.wpActiveEditor : 'content';
        const editor = window.tinymce.get(activeId) || window.tinymce.activeEditor;
        if (editor && editor.id && editor.id !== activeId) {
            try {
                return window.tinymce.get(activeId) || editor;
            } catch (_) {
                return editor;
            }
        }
        return editor || null;
    }

    function deriveSuggestedInnerHtml(targetTag, suggestedText) {
        const html = (typeof suggestedText === 'string') ? suggestedText : '';
        if (!html) { return ''; }
        try {
            const tmp = document.createElement('div');
            tmp.innerHTML = html;
            if (tmp.childElementCount === 1 && targetTag) {
                const child = tmp.firstElementChild;
                if (child && child.tagName && child.tagName.toLowerCase() === String(targetTag).toLowerCase()) {
                    return child.innerHTML;
                }
            }
            const textOnly = tmp.textContent || tmp.innerText;
            if (textOnly && !/<\w+/i.test(html)) {
                return textOnly;
            }
            return tmp.innerHTML || textOnly || html;
        } catch (_) {
            return html;
        }
    }

    function countTagOccurrences(html, tagName) {
        if (!html) { return 0; }
        const re = new RegExp('<\\s*' + tagName + '\\b', 'gi');
        const matches = String(html).match(re);
        return matches ? matches.length : 0;
    }

    function suggestionNeedsBlockReplace(html) {
        if (!html) { return false; }
        const str = String(html);
        if (/<\s*(ul|ol|table|h1|h2|h3|h4|h5|h6)\b/i.test(str)) { return true; }
        return countTagOccurrences(str, 'p') > 1;
    }

    function parseSuggestedBlocks(html) {
        if (!html || !window.wp || !wp.blocks) { return []; }
        try {
            if (typeof wp.blocks.rawHandler === 'function') {
                const blocks = wp.blocks.rawHandler({ HTML: html });
                if (Array.isArray(blocks) && blocks.length) { return blocks; }
            }
        } catch (_) { /* fallback to parse */ }
        try {
            if (typeof wp.blocks.parse === 'function') {
                const blocks = wp.blocks.parse(html);
                if (Array.isArray(blocks) && blocks.length) { return blocks; }
            }
        } catch (_) { /* noop */ }
        return [];
    }

    function normalizeStructureType(type) {
        const t = String(type || '').toLowerCase();
        if (t === 'ordered_list') { return 'numbered_list'; }
        if (t === 'unordered_list' || t === 'bullet_list' || t === 'bullets') { return 'bulleted_list'; }
        if (t === 'bulleted_list' || t === 'numbered_list' || t === 'table' || t === 'paragraph') {
            return t;
        }
        return 'paragraph';
    }

    function formatStructureTypeLabel(type) {
        switch (normalizeStructureType(type)) {
            case 'bulleted_list':
                return (window.wp && wp.i18n ? wp.i18n.__('Bulleted list', 'aeo-content-plugin') : 'Bulleted list');
            case 'numbered_list':
                return (window.wp && wp.i18n ? wp.i18n.__('Numbered list', 'aeo-content-plugin') : 'Numbered list');
            case 'table':
                return (window.wp && wp.i18n ? wp.i18n.__('Table', 'aeo-content-plugin') : 'Table');
            default:
                return (window.wp && wp.i18n ? wp.i18n.__('Paragraph', 'aeo-content-plugin') : 'Paragraph');
        }
    }

    function isManualImprovement(item) {
        if (!item) { return false; }
        const hasSuggested = !!((item.suggestedHtml && item.suggestedHtml.trim()) || (item.suggestedText && item.suggestedText.trim()));
        return !hasSuggested;
    }

    function getSuggestedMarkup(item) {
        if (item && typeof item.suggestedHtml === 'string' && item.suggestedHtml.trim()) {
            return item.suggestedHtml;
        }
        if (item && typeof item.suggestedText === 'string') {
            return item.suggestedText;
        }
        return '';
    }

    function resolveTargetTagFromMatch(match) {
        if (!match || !match.node) { return ''; }
        try {
            if (match.node.querySelector) {
                const inner = match.node.querySelector('p, li, blockquote, h1, h2, h3, h4, h5, h6');
                if (inner && inner.tagName) {
                    return inner.tagName;
                }
            }
        } catch (_) { /* noop */ }
        return match.node.tagName || '';
    }

    function copyHtmlToClipboard(text) {
        if (!text) { return Promise.reject(new Error('empty')); }
        if (navigator.clipboard && window.isSecureContext) {
            return navigator.clipboard.writeText(text);
        }
        return new Promise(function (resolve, reject) {
            try {
                const textarea = document.createElement('textarea');
                textarea.value = text;
                textarea.setAttribute('readonly', 'readonly');
                textarea.style.position = 'fixed';
                textarea.style.top = '-9999px';
                document.body.appendChild(textarea);
                textarea.select();
                const ok = document.execCommand('copy');
                document.body.removeChild(textarea);
                if (ok) { resolve(); } else { reject(new Error('copy-failed')); }
            } catch (err) {
                reject(err);
            }
        });
    }

    function copySuggestedHtml(item, btn) {
        const html = (item && typeof item.suggestedHtml === 'string' && item.suggestedHtml.trim())
            ? item.suggestedHtml
            : (item && item.suggestedText ? item.suggestedText : '');
        if (!html) {
            showToast((window.wp && wp.i18n ? wp.i18n.__('Nothing to copy.', 'aeo-content-plugin') : 'Nothing to copy.'), 'error');
            return;
        }
        copyHtmlToClipboard(html).then(function () {
            if (btn) {
                const original = btn.textContent;
                btn.textContent = (window.wp && wp.i18n ? wp.i18n.__('Copied', 'aeo-content-plugin') : 'Copied');
                setTimeout(function () { btn.textContent = original; }, 1200);
            }
            showToast((window.wp && wp.i18n ? wp.i18n.__('HTML copied.', 'aeo-content-plugin') : 'HTML copied.'), 'success');
        }).catch(function () {
            showToast((window.wp && wp.i18n ? wp.i18n.__('Copy failed. Please try again.', 'aeo-content-plugin') : 'Copy failed. Please try again.'), 'error');
        });
    }

    function markImprovementApplied(item) {
        resolveImprovement(item.id, 'apply').then(function () {
            fetchImprovements().then(function () { decorateEditorSurface(); });
            showToast((window.wp && wp.i18n ? wp.i18n.__('Marked applied.', 'aeo-content-plugin') : 'Marked applied.'), 'success');
        });
    }

    function requestClose() {
        if (isAnalyzing || isLoading) {
            showToast(
                (window.wp && wp.i18n
                    ? wp.i18n.__('Please wait until loading or analysis is complete before closing.', 'aeo-content-plugin')
                    : 'Please wait until loading or analysis is complete before closing.'),
                'error'
            );
            return;
        }
        if (isDirty) {
            const msg = (window.wp && wp.i18n ? wp.i18n.__('You have unsaved changes. Discard them?', 'aeo-content-plugin') : 'You have unsaved changes. Discard them?');
            if (!window.confirm(msg)) { return; }
        }
        closeModal();
    }

    function styleButtonForGutenberg() {
        if (!fabBtn) { return; }
        try {
            fabBtn.classList.remove('button', 'button-primary');
            fabBtn.classList.add('components-button', 'is-primary', 'is-compact', 'has-text');
        } catch (_) { /* noop */ }
    }

    function styleButtonForClassic() {
        if (!fabBtn) { return; }
        try {
            fabBtn.classList.remove('components-button', 'is-primary', 'is-compact', 'has-text');
            if (!fabBtn.classList.contains('button')) { fabBtn.classList.add('button'); }
            if (!fabBtn.classList.contains('button-primary')) { fabBtn.classList.add('button-primary'); }
        } catch (_) { /* noop */ }
    }

    // Try to move button into Gutenberg edit post header
    function findGutenbergHeaderInsertionPoint() {
        // Prefer settings area (right side). Then toolbar. Support legacy and new classnames.
        const selectors = [
            '.editor-header__settings', // new
            '.edit-post-header__settings', // legacy
            '.editor-header__toolbar', // new
            '.edit-post-header__toolbar' // legacy
        ];
        for (let i = 0; i < selectors.length; i++) {
            const el = document.querySelector(selectors[i]);
            if (el) return el;
        }
        // Fallback: search within the header container
        const header = document.querySelector('.editor-header.edit-post-header');
        if (header) {
            const right = header.querySelector('.editor-header__settings, .edit-post-header__settings');
            if (right) return right;
            const left = header.querySelector('.editor-header__toolbar, .edit-post-header__toolbar');
            if (left) return left;
            return header; // last resort, still same row container
        }
        return document.querySelector('.interface-interface-skeleton__header') || null;
    }

    function mountButtonInGutenbergHeader() {
        if (!isGutenbergEnv() || !buttonContainer) return false;
        const target = findGutenbergHeaderInsertionPoint();
        if (!target) return false;

        // Wrap for header placement if not already
        let wrap = document.getElementById('aeo-toolbar-btn-wrap');
        if (!wrap) {
            wrap = document.createElement('div');
            wrap.id = 'aeo-toolbar-btn-wrap';
            wrap.className = 'aeo-toolbar-btn-wrap';
        }

        // Move the button into the header wrapper
        if (fabBtn && wrap && fabBtn.parentNode !== wrap) {
            wrap.appendChild(fabBtn);
        }
        // Insert wrapper into header if not already present
        if (wrap.parentNode !== target) {
            target.appendChild(wrap);
        }

        // Hide original floating container
        if (buttonContainer) {
            buttonContainer.style.display = 'none';
        }

        styleButtonForGutenberg();

        return true;
    }

    function findClassicToolbarInsertionPoint() {
        const toolbar = document.querySelector('.wp-editor-tools');
        if (toolbar) { return toolbar; }
        const titleDiv = document.getElementById('titlediv');
        if (titleDiv) { return titleDiv; }
        return document.getElementById('wpbody-content') || null;
    }

    function mountButtonInClassicToolbar() {
        if (isGutenbergEnv() || !fabBtn) return false;
        const target = findClassicToolbarInsertionPoint();
        if (!target) return false;

        let wrap = document.getElementById('aeo-classic-btn-wrap');
        if (!wrap) {
            wrap = document.createElement('div');
            wrap.id = 'aeo-classic-btn-wrap';
            wrap.className = 'aeo-classic-btn-wrap';
        }

        if (fabBtn.parentNode !== wrap) {
            wrap.appendChild(fabBtn);
        }
        if (!target.contains(wrap)) {
            target.appendChild(wrap);
        }

        if (buttonContainer) {
            buttonContainer.style.display = 'none';
        }

        styleButtonForClassic();
        wrap.style.display = hasPostId ? 'flex' : 'none';
        return true;
    }

    function mountButtonForEditor() {
        if (isGutenbergEnv()) {
            return mountButtonInGutenbergHeader();
        }
        return mountButtonInClassicToolbar();
    }

    // Attempt immediately and then retry a few times as the editor mounts
    (function attemptButtonMount() {
        let tries = 0;
        const maxTries = HEADER_MOUNT_MAX_RETRIES; // ~8s total
        const iv = setInterval(function () {
            tries++;
            if (mountButtonForEditor() || tries >= maxTries) {
                clearInterval(iv);
            }
        }, 200);
        // Also a one-shot immediate try
        mountButtonForEditor();
    })();

    // Clear overlays during save to avoid any accidental persistence and re-decorate post-save
    (function subscribeToPostSaveLifecycle() {
        if (!isGutenbergEnv()) return;
        try {
            const selEditor = wp.data.select('core/editor');
            let lastSaving = false;
            wp.data.subscribe(function () {
                let saving = false;
                try {
                    saving = !!(selEditor.isSavingPost() || selEditor.isAutosavingPost());
                } catch (_) { saving = false; }
                if (saving && !lastSaving) {
                    try { clearDecorations(); } catch (_) { }
                }
                if (!saving && lastSaving) {
                    try {
                        sanitizeParagraphContentAttributes();
                        fetchImprovements().then(function () { try { decorateEditorSurface(); } catch (_) { } });
                    } catch (_) { /* noop */ }
                }
                lastSaving = saving;
            });
        } catch (_) { /* best-effort */ }
    })();

    function initAnalyzeSaveGate() {
        if (isGutenbergEnv()) {
            if (!(window.wp && wp.data && wp.data.subscribe)) { return; }
            let lastKey = '';
            try { lastKey = getGutenbergSaveGateKey(); } catch (_) { lastKey = ''; }
            wp.data.subscribe(function () {
                const key = getGutenbergSaveGateKey();
                if (key !== lastKey) {
                    lastKey = key;
                    updateAnalyzeButton();
                }
            });
            return;
        }
        ensureClassicBaseline();
        const titleInput = document.getElementById('title');
        if (titleInput) {
            titleInput.addEventListener('input', function () { updateAnalyzeButton(); });
        }
        const textarea = getClassicTextarea();
        if (textarea) {
            textarea.addEventListener('input', function () { updateAnalyzeButton(); });
        }
        const attachEditor = function (editor) {
            if (!editor || typeof editor.on !== 'function') { return; }
            editor.on('change keyup NodeChange input', function () { updateAnalyzeButton(); });
        };
        try {
            attachEditor(getClassicTinyMCEEditor());
            if (window.tinymce && typeof window.tinymce.on === 'function') {
                window.tinymce.on('AddEditor', function (e) { attachEditor(e.editor); });
            }
        } catch (_) { /* noop */ }
    }

    // Show/hide button based on postId presence
    function updateButtonVisibility() {
        const wrap = document.getElementById('aeo-toolbar-btn-wrap');
        const classicWrap = document.getElementById('aeo-classic-btn-wrap');
        const container = document.getElementById('aeo-button-container');
        const visible = !!hasPostId;
        if (wrap) wrap.style.display = visible ? '' : 'none';
        if (classicWrap) classicWrap.style.display = visible ? 'flex' : 'none';
        if (container) {
            const shouldShowFab = isGutenbergEnv();
            container.classList.toggle('aeo-visible', visible && shouldShowFab);
            container.style.display = shouldShowFab ? '' : 'none';
        }
    }
    updateButtonVisibility();
    initAnalyzeSaveGate();

    let classicRefreshRaf = null;
    function scheduleClassicRefresh() {
        if (classicRefreshRaf) cancelAnimationFrame(classicRefreshRaf);
        classicRefreshRaf = requestAnimationFrame(function () {
            classicRefreshRaf = null;
            try { setEditorMode(detectClassicEditorMode()); } catch (_) { }
            try { kickDecorations(); } catch (_) { }
        });
    }

    function observeClassicModeChanges() {
        if (isGutenbergEnv()) { return; }
        const wrap = document.getElementById('wp-content-wrap');
        if (!wrap || typeof MutationObserver === 'undefined') { return; }
        let lastMode = detectClassicEditorMode();
        const observer = new MutationObserver(function () {
            const nextMode = detectClassicEditorMode();
            if (nextMode !== lastMode) {
                lastMode = nextMode;
                scheduleClassicRefresh();
            }
        });
        observer.observe(wrap, { attributes: true, attributeFilter: ['class'] });
        window.addEventListener('tinymce-editor-init', function () {
            scheduleClassicRefresh();
        });
    }
    observeClassicModeChanges();

    (function installClassicEditorListeners() {
        if (isGutenbergEnv()) { return; }
        if (installClassicEditorListeners._done) return;
        installClassicEditorListeners._done = true;

        function bindTinyMceEditor(editor) {
            if (!editor || editor._aeoClassicBound) return;
            editor._aeoClassicBound = true;
            const events = ['Change', 'Undo', 'Redo', 'KeyUp', 'SetContent', 'Paste'];
            events.forEach(function (evt) {
                try { editor.on(evt, scheduleClassicRefresh); } catch (_) { }
            });
            try { editor.on('SwitchMode', scheduleClassicRefresh); } catch (_) { }
        }

        function bindExistingEditors() {
            if (!window.tinymce || !Array.isArray(window.tinymce.editors)) { return; }
            window.tinymce.editors.forEach(bindTinyMceEditor);
        }

        bindExistingEditors();

        document.addEventListener('tinymce-editor-init', function (event) {
            let editor = null;
            if (event && event.detail && event.detail.editor) {
                editor = event.detail.editor;
            } else if (event && event.target && event.target.id && window.tinymce) {
                editor = window.tinymce.get(event.target.id);
            }
            bindTinyMceEditor(editor);
            scheduleClassicRefresh();
        });

        setTimeout(bindExistingEditors, 1500);

        const textarea = getClassicTextarea();
        if (textarea && !textarea._aeoClassicBound) {
            textarea._aeoClassicBound = true;
            ['input', 'change', 'blur'].forEach(function (evt) {
                textarea.addEventListener(evt, scheduleClassicRefresh);
            });
        }
    })();

    // ===== Header pending count badge =====
    let countBadge = null;
    function ensureCountBadge() {
        if (countBadge) return countBadge;
        const existing = document.getElementById('aeo-count-badge') || (fabBtn ? fabBtn.querySelector('.aeo-count-badge') : null);
        if (existing) {
            countBadge = existing;
            return countBadge;
        }
        countBadge = document.createElement('span');
        countBadge.className = 'aeo-count-badge is-empty';
        countBadge.setAttribute('aria-label', (window.wp && wp.i18n ? wp.i18n.__('Pending suggestions', 'aeo-content-plugin') : 'Pending suggestions'));
        if (fabBtn) {
            fabBtn.appendChild(countBadge);
        } else if (buttonContainer) {
            buttonContainer.appendChild(countBadge);
        } else {
            // fallback append to root
            root.appendChild(countBadge);
        }
        return countBadge;
    }

    function updateCountBadge() {
        const badge = ensureCountBadge();
        if (!badge) { return; }
        const total = Array.isArray(impr.items) ? impr.items.length : 0;
        if (total > 0) {
            badge.textContent = String(total);
            badge.classList.remove('is-empty');
            badge.title = (window.wp && wp.i18n ? wp.i18n.__('Pending suggestions', 'aeo-content-plugin') : 'Pending suggestions');
        } else {
            badge.textContent = '';
            badge.classList.add('is-empty');
            badge.removeAttribute('title');
        }
    }

    // Initial badge render (actual patching occurs after functions are declared)
    try { updateCountBadge(); } catch (_) { }

    // Wire events
    if (fabBtn) {
        fabBtn.addEventListener('click', openModal);
    }
    if (closeBtn) {
        closeBtn.addEventListener('click', requestClose);
    }
    if (cancelBtn) {
        cancelBtn.addEventListener('click', requestClose);
    }
    if (backdrop) {
        backdrop.addEventListener('click', requestClose);
    }
    document.addEventListener('keydown', function (ev) {
        if (ev.key === 'Escape' && modal && modal.classList.contains('aeo-open')) {
            if (!isAnalyzing && !isLoading) { requestClose(); }
        }
    });

    // Tabs
    function activateTab(which) {
        const isOverview = which === 'overview';
        const isFaqs = which === 'faqs';
        const isDebug = which === 'debug';

        if (tabBtnOverview) {
            tabBtnOverview.classList.toggle('is-active', isOverview);
            tabBtnOverview.setAttribute('aria-selected', String(isOverview));
        }
        if (tabBtnFaqs) {
            tabBtnFaqs.classList.toggle('is-active', isFaqs);
            tabBtnFaqs.setAttribute('aria-selected', String(isFaqs));
        }
        if (tabBtnDebug) {
            tabBtnDebug.classList.toggle('is-active', isDebug);
            tabBtnDebug.setAttribute('aria-selected', String(isDebug));
        }

        if (sectionOverview) sectionOverview.classList.toggle('is-active', isOverview);
        if (sectionFaqs) sectionFaqs.classList.toggle('is-active', isFaqs);
        if (sectionDebug) sectionDebug.classList.toggle('is-active', isDebug);

        // When debug tab is activated, log frontend matching state to console
        if (isDebug) {
            try {
                console.log('[AEO] Debug Tab Activated');
                const segments = segmentEditorContent();
                console.log('[AEO] Frontend Segments:', segments);
                console.log('[AEO] Current Improvements:', impr.items);
            } catch (_) { }
        }
    }
    if (tabBtnOverview) tabBtnOverview.addEventListener('click', function () { activateTab('overview'); });
    if (tabBtnFaqs) tabBtnFaqs.addEventListener('click', function () { activateTab('faqs'); });
    if (tabBtnDebug) tabBtnDebug.addEventListener('click', function () { activateTab('debug'); });

    if (debugCopyBtn && debugLogEl) {
        debugCopyBtn.addEventListener('click', function () {
            debugLogEl.select();
            document.execCommand('copy');
            const originalText = debugCopyBtn.textContent;
            debugCopyBtn.textContent = 'Copied!';
            setTimeout(function () { debugCopyBtn.textContent = originalText; }, 2000);
        });
    }

    // Overview count (live)
    function updateOverviewCount() {
        if (!overviewEl || !overviewCount) return;
        const len = (overviewEl.value || '').length;
        overviewCount.textContent = String(len);
    }
    if (overviewEl) {
        overviewEl.addEventListener('input', function () {
            updateOverviewCount();
            updateDirtyFromCurrent();
        });
        updateOverviewCount();
    }

    // FAQ helpers
    function clearErrorOnInput(el) {
        try {
            if (!el._aeoClearErrorAttached) {
                el.addEventListener('input', function () {
                    el.removeAttribute('aria-invalid');
                    if (el.style) { el.style.removeProperty('border-color'); }
                    const parent = el.closest && el.closest('.aeo-faq-item');
                    if (parent) { parent.classList.remove('has-error'); }
                });
                el._aeoClearErrorAttached = true;
            }
        } catch (_) { /* noop */ }
    }
    function updateFaqsEmpty() {
        if (!faqsList || !faqsEmpty) return;
        const hasItems = !!faqsList.querySelector('.aeo-faq-item');
        faqsEmpty.style.display = hasItems ? 'none' : 'block';
    }
    function createFaqItem(qVal, aVal) {
        const wrap = document.createElement('div');
        wrap.className = 'aeo-faq-item';
        wrap.innerHTML = [
            '<div class="aeo-faq-head" tabindex="0">',
            '<span class="aeo-faq-title">' + (qVal ? escapeHtml(qVal) : (window.wp && wp.i18n ? wp.i18n.__('New question', 'aeo-content-plugin') : 'New question')) + '</span>',
            '<div class="aeo-faq-actions">',
            '<button type="button" class="button-link aeo-faq-edit">' + (window.wp && wp.i18n ? wp.i18n.__('Edit', 'aeo-content-plugin') : 'Edit') + '</button>',
            '<button type="button" class="button-link aeo-faq-delete">' + (window.wp && wp.i18n ? wp.i18n.__('Delete', 'aeo-content-plugin') : 'Delete') + '</button>',
            '</div>',
            '</div>',
            '<div class="aeo-faq-fields">',
            '<label class="aeo-label">' + (window.wp && wp.i18n ? wp.i18n.__('Question', 'aeo-content-plugin') : 'Question') + '</label>',
            '<input type="text" class="aeo-input aeo-faq-q" value="' + (qVal ? escapeHtmlAttr(qVal) : '') + '" />',
            '<div class="aeo-faq-answer-head">',
            '<label class="aeo-label aeo-label-spaced">' + (window.wp && wp.i18n ? wp.i18n.__('Answer', 'aeo-content-plugin') : 'Answer') + '</label>',
            '<button type="button" class="button aeo-faq-generate" disabled>',
            '<span class="aeo-faq-generate-label">' + (window.wp && wp.i18n ? wp.i18n.__('Generate answer', 'aeo-content-plugin') : 'Generate answer') + '</span>',
            '<span class="aeo-faq-generate-spinner" aria-hidden="true"></span>',
            '</button>',
            '</div>',
            '<textarea class="aeo-input aeo-faq-a" rows="4">' + (aVal ? escapeHtml(aVal) : '') + '</textarea>',
            '<p class="aeo-helper aeo-faq-generate-error" role="alert" style="display:none;"></p>',
            '</div>'
        ].join('');

        const head = wrap.querySelector('.aeo-faq-head');
        const title = wrap.querySelector('.aeo-faq-title');
        const edit = wrap.querySelector('.aeo-faq-edit');
        const del = wrap.querySelector('.aeo-faq-delete');
        const q = wrap.querySelector('.aeo-faq-q');
        const a = wrap.querySelector('.aeo-faq-a');
        const genBtn = wrap.querySelector('.aeo-faq-generate');
        const genLabel = wrap.querySelector('.aeo-faq-generate-label');
        const genError = wrap.querySelector('.aeo-faq-generate-error');
        let isGenerating = false;

        function setGenerateError(msg) {
            if (!genError) return;
            if (msg) {
                genError.textContent = String(msg);
                genError.style.display = 'block';
            } else {
                genError.textContent = '';
                genError.style.display = 'none';
            }
        }

        function updateGenerateButton() {
            if (!genBtn) return;
            const hasQuestion = !!(q && (q.value || '').trim());
            const licenseActive = !!(window.aeoEditor && window.aeoEditor.env && window.aeoEditor.env.licenseActive);
            const disabled = !licenseActive || !hasQuestion || isGenerating;
            if (disabled) {
                genBtn.setAttribute('disabled', 'disabled');
            } else {
                genBtn.removeAttribute('disabled');
            }
        }

        function setGeneratingState(loading) {
            isGenerating = !!loading;
            if (!genBtn) return;
            genBtn.classList.toggle('is-loading', isGenerating);
            if (genLabel) {
                genLabel.textContent = isGenerating
                    ? (window.wp && wp.i18n ? wp.i18n.__('Generating...', 'aeo-content-plugin') : 'Generating...')
                    : (window.wp && wp.i18n ? wp.i18n.__('Generate answer', 'aeo-content-plugin') : 'Generate answer');
            }
            updateGenerateButton();
        }

        function setOpen(open) {
            wrap.classList.toggle('is-open', open);
            // Reflect expanded state for accessibility
            if (head) { head.setAttribute('aria-expanded', String(open)); }
            if (open) { q && q.focus(); }
        }
        function syncTitle() {
            title.textContent = (q.value || '').trim() || (window.wp && wp.i18n ? wp.i18n.__('New question', 'aeo-content-plugin') : 'New question');
        }

        head.addEventListener('click', function (e) {
            if (e.target === del || e.target === edit) return;
            setOpen(!wrap.classList.contains('is-open'));
        });
        head.addEventListener('keydown', function (e) { if (e.key === 'Enter' || e.key === ' ') { e.preventDefault(); setOpen(!wrap.classList.contains('is-open')); } });
        if (edit) {
            edit.addEventListener('click', function (e) {
                e.preventDefault();
                const willOpen = !wrap.classList.contains('is-open');
                setOpen(willOpen);
                if (willOpen && q) { q.focus(); }
            });
        }
        del.addEventListener('click', function () { wrap.remove(); updateFaqsEmpty(); updateDirtyFromCurrent(); });
        q.addEventListener('input', function () {
            syncTitle();
            setGenerateError('');
            updateGenerateButton();
            updateDirtyFromCurrent();
        });
        if (a) {
            a.addEventListener('input', function () {
                setGenerateError('');
                updateDirtyFromCurrent();
            });
        }
        if (genBtn) {
            genBtn.addEventListener('click', function (e) {
                e.preventDefault();
                if (isGenerating) return;
                const questionText = q ? (q.value || '').trim() : '';
                if (!questionText) {
                    setGenerateError(window.wp && wp.i18n
                        ? wp.i18n.__('Please enter a question first.', 'aeo-content-plugin')
                        : 'Please enter a question first.');
                    updateGenerateButton();
                    return;
                }
                if (!(window.aeoEditor && window.aeoEditor.env && window.aeoEditor.env.licenseActive)) {
                    setGenerateError(window.wp && wp.i18n
                        ? wp.i18n.__('Activate your license to generate answers.', 'aeo-content-plugin')
                        : 'Activate your license to generate answers.');
                    updateGenerateButton();
                    return;
                }
                if (a && (a.value || '').trim() !== '') {
                    const confirmMsg = (window.wp && wp.i18n)
                        ? wp.i18n.__('Replace the current answer with a generated one?', 'aeo-content-plugin')
                        : 'Replace the current answer with a generated one?';
                    if (!window.confirm(confirmMsg)) { return; }
                }

                setGenerateError('');
                setGeneratingState(true);

                const params = new URLSearchParams();
                params.append('action', 'aeo_generate_faq_answer');
                params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.editor ? window.aeoEditor.env.nonces.editor : '');
                params.append('post_id', String(window.aeoEditor.env.postId || 0));
                params.append('question', questionText);

                fetch(cfg.ajax_url, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
                    body: params.toString(),
                    credentials: 'same-origin'
                })
                    .then(function (r) { return r.json(); })
                    .then(function (resp) {
                        if (!resp || !resp.success || !resp.data) {
                            const msg = resp && resp.data && resp.data.message
                                ? resp.data.message
                                : (window.wp && wp.i18n ? wp.i18n.__('Unable to generate answer.', 'aeo-content-plugin') : 'Unable to generate answer.');
                            throw new Error(msg);
                        }
                        const answerText = (resp.data.answer || '').trim();
                        if (!answerText) {
                            throw new Error(window.wp && wp.i18n
                                ? wp.i18n.__('Empty answer returned. Try again.', 'aeo-content-plugin')
                                : 'Empty answer returned. Try again.');
                        }
                        if (a) { a.value = answerText; }
                        updateDirtyFromCurrent();
                    })
                    .catch(function (err) {
                        const msg = (err && err.message)
                            ? err.message
                            : (window.wp && wp.i18n ? wp.i18n.__('Unable to generate answer.', 'aeo-content-plugin') : 'Unable to generate answer.');
                        setGenerateError(msg);
                    })
                    .finally(function () {
                        setGeneratingState(false);
                    });
            });
        }
        // Clear field-level error visuals on input
        if (q) { clearErrorOnInput(q); }
        if (a) { clearErrorOnInput(a); }
        updateGenerateButton();

        // Drag-and-drop reordering via header as handle
        try {
            head.setAttribute('draggable', 'true');
            head.addEventListener('dragstart', function (e) {
                if ((e.target && e.target.closest && e.target.closest('.aeo-faq-delete'))) { return; }
                wrap.classList.add('dragging');
                if (e.dataTransfer) {
                    e.dataTransfer.effectAllowed = 'move';
                    // setData required by some browsers to start the drag
                    e.dataTransfer.setData('text/plain', 'aeo-faq-item');
                }
            });
            head.addEventListener('dragend', function () {
                wrap.classList.remove('dragging');
                updateDirtyFromCurrent();
            });
        } catch (_) { /* noop */ }

        return wrap;
    }
    function refreshFaqGenerateButtons(forceDisabled) {
        if (!faqsList) return;
        const lockDisabled = !!forceDisabled;
        const licenseActive = !!(window.aeoEditor && window.aeoEditor.env && window.aeoEditor.env.licenseActive);
        faqsList.querySelectorAll('.aeo-faq-item').forEach(function (item) {
            const q = item.querySelector('.aeo-faq-q');
            const genBtn = item.querySelector('.aeo-faq-generate');
            if (!genBtn) return;
            const hasQuestion = !!(q && (q.value || '').trim());
            const isLoading = genBtn.classList.contains('is-loading');
            const disabled = lockDisabled || !licenseActive || !hasQuestion || isLoading;
            if (disabled) {
                genBtn.setAttribute('disabled', 'disabled');
            } else {
                genBtn.removeAttribute('disabled');
            }
        });
    }
    function escapeHtml(str) { return String(str).replace(/[&<>"']/g, function (m) { return ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', '\'': '&#39;' }[m]); }); }
    function escapeHtmlAttr(str) { return escapeHtml(str).replace(/"/g, '&quot;'); }

        if (addFaqBtn && faqsList) {
            addFaqBtn.addEventListener('click', function () {
                const node = createFaqItem('', '');
                faqsList.appendChild(node);
                updateFaqsEmpty();
                node.classList.add('is-open');
                const q = node.querySelector('.aeo-faq-q');
                if (q) q.focus();
                updateDirtyFromCurrent();
                refreshFaqGenerateButtons();
            });
        }
        updateFaqsEmpty();

    // Helper to find insertion point while dragging
    function getDragAfterElement(container, y) {
        const elements = Array.prototype.slice.call(container.querySelectorAll('.aeo-faq-item:not(.dragging)'));
        let closest = { offset: Number.NEGATIVE_INFINITY, element: null };
        for (let i = 0; i < elements.length; i++) {
            const el = elements[i];
            const rect = el.getBoundingClientRect();
            const offset = y - rect.top - rect.height / 2;
            if (offset < 0 && offset > closest.offset) {
                closest = { offset, element: el };
            }
        }
        return closest.element;
    }

    // List-level dragover to reorder items on the fly
    if (faqsList) {
        faqsList.addEventListener('dragover', function (e) {
            e.preventDefault();
            const dragging = faqsList.querySelector('.aeo-faq-item.dragging');
            if (!dragging) return;
            const after = getDragAfterElement(faqsList, e.clientY);
            if (after == null) {
                faqsList.appendChild(dragging);
            } else if (after !== dragging) {
                faqsList.insertBefore(dragging, after);
            }
        });
    }

    // License banner and gating
    function updateLicenseBanner() {
        if (!licenseBanner) return;
        const active = !!window.aeoEditor.env.licenseActive;
        if (!active) {
            const url = 'admin.php?page=aeo-content-plugin';
            licenseBanner.classList.add('is-visible');
            const remaining = trialRunsRemaining();
            const limit = trialLimit();
            const notice = getTrialState().notice;
            let message;
            if (remaining > 0) {
                const fallback = 'Free preview active. ' + remaining + ' of ' + (limit || 10) + ' runs remaining. Activate your license for unlimited access.';
                if (window.wp && wp.i18n && wp.i18n.__) {
                    const str = wp.i18n.__('Free preview active. %1$d of %2$d runs remaining. Activate your license for unlimited access.', 'aeo-content-plugin');
                    if (window.wp.i18n && typeof window.wp.i18n.sprintf === 'function') {
                        message = notice || window.wp.i18n.sprintf(str, remaining, limit || 10);
                    } else {
                        message = notice || str.replace('%1$d', remaining).replace('%2$d', limit || 10);
                    }
                } else {
                    message = notice || fallback;
                }
            } else {
                message = notice || ((window.wp && wp.i18n ? wp.i18n.__('Free preview exhausted. Activate your license to continue.', 'aeo-content-plugin') : 'Free preview exhausted. Activate your license to continue.'));
            }
            licenseBanner.innerHTML = '<p><strong>' + (remaining > 0
                ? (window.wp && wp.i18n ? wp.i18n.__('Free preview', 'aeo-content-plugin') : 'Free preview')
                : (window.wp && wp.i18n ? wp.i18n.__('License inactive', 'aeo-content-plugin') : 'License inactive')) + '.</strong> '
                + message + ' <a href="' + url + '">' + (window.wp && wp.i18n ? wp.i18n.__('Open settings', 'aeo-content-plugin') : 'Open settings') + '</a></p>';
            if (overviewEl) overviewEl.setAttribute('readonly', 'readonly');
            if (faqsList) {
                faqsList.querySelectorAll('input,textarea,button').forEach(function (el) { el.setAttribute('disabled', 'disabled'); });
                if (addFaqBtn) addFaqBtn.setAttribute('disabled', 'disabled');
            }
            if (saveBtn) { saveBtn.setAttribute('disabled', 'disabled'); saveBtn.style.display = 'none'; }
            if (analyzeBtn) {
                if (remaining > 0) {
                    analyzeBtn.removeAttribute('disabled');
                } else {
                    analyzeBtn.setAttribute('disabled', 'disabled');
                }
            }
            if (readonlyOverlay) readonlyOverlay.style.display = 'block';
            if (licenseCta) licenseCta.style.display = 'inline-block';
        } else {
            licenseBanner.classList.remove('is-visible');
            if (readonlyOverlay) readonlyOverlay.style.display = 'none';
            if (licenseCta) licenseCta.style.display = 'none';
            if (saveBtn) { saveBtn.style.display = ''; }
        }
        updateSaveButton();
    }
    updateLicenseBanner();

    // Listen for license activation changes (cross-tab or same tab)
    try {
        if ('BroadcastChannel' in window) {
            const ch = new BroadcastChannel('aeo_license');
            ch.onmessage = function (ev) {
                if (!ev || !ev.data) return;
                if (ev.data.type === 'license-status' && typeof ev.data.active === 'boolean') {
                    window.aeoEditor.env.licenseActive = !!ev.data.active;
                    updateLicenseBanner();
                    updateAnalyzeButton();
                    updateAnalyzeLabel();
                    setFormEnabled(true);
                }
            };
            // Also listen for usage refresh notifications and fetch server-truth status
            try {
                const usageCh = new BroadcastChannel('aeo_usage');
                usageCh.onmessage = function (ev) {
                    if (ev && ev.data && ev.data.type === 'refresh') {
                        try { refreshUsageStatus(); } catch (_) { /* noop */ }
                    }
                };
            } catch (_) { /* noop */ }
        }
        window.addEventListener('storage', function (e) {
            if (e.key === 'aeo_license_status' && typeof e.newValue === 'string') {
                const active = (e.newValue === 'active');
                window.aeoEditor.env.licenseActive = active;
                updateLicenseBanner();
                updateAnalyzeButton();
                updateAnalyzeLabel();
                setFormEnabled(true);
            }
            if (e && e.key === 'aeo_usage_refresh') {
                try { refreshUsageStatus(); } catch (_) { /* noop */ }
            }
        });
    } catch (_) { /* best-effort */ }

    // Enable/disable form controls (used during load); license gating reapplied afterwards
    function setFormEnabled(enabled) {
        const enable = !!enabled;
        if (overviewEl) {
            if (enable) { overviewEl.removeAttribute('readonly'); }
            else { overviewEl.setAttribute('readonly', 'readonly'); }
        }
        if (faqsList) {
            faqsList.querySelectorAll('input,textarea,button').forEach(function (el) {
                if (enable) { el.removeAttribute('disabled'); }
                else { el.setAttribute('disabled', 'disabled'); }
            });
        }
        if (addFaqBtn) {
            if (enable) { addFaqBtn.removeAttribute('disabled'); }
            else { addFaqBtn.setAttribute('disabled', 'disabled'); }
        }
        if (saveBtn) {
            if (enable) { saveBtn.removeAttribute('disabled'); }
            else { saveBtn.setAttribute('disabled', 'disabled'); }
        }
        if (analyzeBtn) {
            if (enable) { analyzeBtn.removeAttribute('disabled'); }
            else { analyzeBtn.setAttribute('disabled', 'disabled'); }
        }
        // Re-apply license gating after raw enable/disable so license state wins
        updateLicenseBanner();
        updateAnalyzeButton();
        updateAnalyzeLabel();
        refreshFaqGenerateButtons(!enable);
    }

    function getCurrentState() {
        const ov = overviewEl ? (overviewEl.value || '').trim() : '';
        const items = [];
        if (faqsList) {
            faqsList.querySelectorAll('.aeo-faq-item').forEach(function (item) {
                const q = item.querySelector('.aeo-faq-q');
                const a = item.querySelector('.aeo-faq-a');
                const qv = q ? (q.value || '').trim() : '';
                const av = a ? (a.value || '').trim() : '';
                // Keep entries even if empty to reflect user edits; saving will filter later
                items.push({ question: qv, answer: av });
            });
        }
        return { overview: ov, faqs: items };
    }

    function shallowEqualArrays(a, b) {
        if (!Array.isArray(a) || !Array.isArray(b)) return false;
        if (a.length !== b.length) return false;
        for (let i = 0; i < a.length; i++) {
            if (a[i].question !== b[i].question || a[i].answer !== b[i].answer) return false;
        }
        return true;
    }

    function setDirty(val) {
        isDirty = !!val;
        updateSaveButton();
    }

    function isStateEqual(a, b) {
        if (!a || !b) return false;
        return (a.overview === b.overview) && shallowEqualArrays(a.faqs, b.faqs);
    }

    function updateDirtyFromCurrent() {
        const cur = getCurrentState();
        setDirty(!isStateEqual(cur, initialState));
    }

    function updateSaveButton() {
        if (!saveBtn) return;
        const canSave = !!(window.aeoEditor.env.licenseActive && isDirty && !isAnalyzing);
        if (canSave) { saveBtn.removeAttribute('disabled'); }
        else { saveBtn.setAttribute('disabled', 'disabled'); }
    }

    function updateAnalyzeButton() {
        if (!analyzeBtn) return;
        const blockedByBulk = !!window.aeoEditor.env.bulkActive;
        const st = (window.aeoEditor.env.autoStatus || '').toLowerCase();
        const blockedByAuto = (st === 'queued' || st === 'running');
        const saveGate = getAnalyzeSaveGate();
        const blockedByUnsaved = !saveGate.saved;
        const hasAccess = hasTrialAccess();
        const disabled = (!hasAccess) || isAnalyzing || blockedByBulk || blockedByAuto || blockedByUnsaved;
        if (disabled) {
            analyzeBtn.setAttribute('disabled', 'disabled');
            let reason = '';
            if (blockedByUnsaved) {
                reason = saveGate.reason || getAnalyzeSaveRequiredMessage();
            } else if (!hasAccess) {
                reason = trialRunsRemaining() > 0
                    ? ''
                    : (window.wp && wp.i18n ? wp.i18n.__('Free preview exhausted. Activate your license to continue analyzing.', 'aeo-content-plugin') : 'Free preview exhausted. Activate your license to continue analyzing.');
            }
            else if (blockedByBulk) reason = (window.wp && wp.i18n ? wp.i18n.__('Disabled during bulk analysis.', 'aeo-content-plugin') : 'Disabled during bulk analysis.');
            else if (blockedByAuto) reason = (window.wp && wp.i18n ? wp.i18n.__('Automatic analysis in progress.', 'aeo-content-plugin') : 'Automatic analysis in progress.');
            analyzeBtn.title = reason;
        } else {
            analyzeBtn.removeAttribute('disabled');
            if (!window.aeoEditor.env.licenseActive && trialRunsRemaining() > 0) {
                const limit = trialLimit() || 10;
                const remaining = trialRunsRemaining();
                if (window.wp && wp.i18n && typeof wp.i18n.sprintf === 'function') {
                    analyzeBtn.title = wp.i18n.sprintf(wp.i18n.__('Free preview: %1$d of %2$d runs remaining.', 'aeo-content-plugin'), remaining, limit);
                } else {
                    analyzeBtn.title = 'Free preview: ' + remaining + ' of ' + limit + ' runs remaining.';
                }
            } else {
                analyzeBtn.title = '';
            }
        }
    }

    function getAnalyzeBaseLabel() {
        const analyzed = !!(window.aeoEditor && window.aeoEditor.env && window.aeoEditor.env.analyzed);
        if (window.wp && wp.i18n) {
            return analyzed ? wp.i18n.__('Re-analyze', 'aeo-content-plugin') : wp.i18n.__('Analyze', 'aeo-content-plugin');
        }
        return analyzed ? 'Re-analyze' : 'Analyze';
    }
    function updateAnalyzeLabel() {
        if (!analyzeBtn) return;
        analyzeBtn.textContent = getAnalyzeBaseLabel();
    }

    function snapshotState() {
        initialState = getCurrentState();
        setDirty(false);
    }

    function rebuildFaqsList(faqs) {
        if (!faqsList) return;
        faqsList.innerHTML = '';
        (Array.isArray(faqs) ? faqs : []).forEach(function (f) {
            const node = createFaqItem(f.question || '', f.answer || '');
            faqsList.appendChild(node);
        });
        updateFaqsEmpty();
        refreshFaqGenerateButtons(isLoading || isAnalyzing);
    }

    function loadData() {
        if (!hasPostId || !cfg || !cfg.ajax_url) { return; }
        if (isLoading) { return; }
        isLoading = true;
        setFormEnabled(false);
        if (closeBtn) closeBtn.setAttribute('disabled', 'disabled');
        if (cancelBtn) cancelBtn.setAttribute('disabled', 'disabled');
        showOverlay((window.wp && wp.i18n ? wp.i18n.__('Loading…', 'aeo-content-plugin') : 'Loading…'));
        const params = new URLSearchParams();
        params.append('action', 'aeo_editor_read');
        params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.editor ? window.aeoEditor.env.nonces.editor : '');
        params.append('post_id', String(window.aeoEditor.env.postId || 0));
        fetch(cfg.ajax_url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
            body: params.toString(),
            credentials: 'same-origin'
        }).then(function (r) { return r.json(); })
            .then(function (resp) {
                if (!resp || !resp.success || !resp.data) { throw new Error('Failed to load data'); }
                const data = resp.data;
                // Sync env
                window.aeoEditor.env.licenseActive = !!data.license_active;
                window.aeoEditor.env.analyzed = !!data.analyzed;
                window.aeoEditor.env.autoStatus = (data.auto_status || '');
                if (data.trial) {
                    setTrialState(data.trial);
                }
                // Populate
                if (overviewEl) { overviewEl.value = data.overview || ''; updateOverviewCount(); }
                rebuildFaqsList(data.faqs || []);
                if (Array.isArray(data.improvements)) {
                    setImprovementsList(data.improvements);
                }
                // Populate Debug Log
                if (debugLogEl) {
                    if (data.debug_log) {
                        debugLogEl.value = JSON.stringify(data.debug_log, null, 2);
                    } else {
                        debugLogEl.value = 'No debug log available for this post.';
                    }
                }
                // Apply gating and reset dirty baseline
                updateLicenseBanner();
                updateAnalyzeButton();
                updateAnalyzeLabel();
                snapshotState();
            }).catch(function (err) {
                // Minimal surface: log and keep current UI
                console && console.error && console.error('AEO editor load failed:', err);
            }).finally(function () {
                isLoading = false;
                // Re-enable form according to license
                setFormEnabled(true);
                if (!isAnalyzing) { hideOverlay(); }
                if (closeBtn) closeBtn.removeAttribute('disabled');
                if (cancelBtn) cancelBtn.removeAttribute('disabled');
            });
    }

    // ===== Phase 8: Save Flow =====
    function ensureErrorContainer() {
        if (!modal) return null;
        let err = modal.querySelector('#aeo-modal-error');
        if (!err) {
            err = document.createElement('div');
            err.id = 'aeo-modal-error';
            err.setAttribute('role', 'alert');
            err.style.display = 'none';
            err.style.margin = '12px 16px 0';
            err.style.padding = '8px 12px';
            err.style.background = '#f6f7f7';
            err.style.borderLeft = '4px solid #d63638';
            const body = modal.querySelector('.aeo-modal-body');
            const header = modal.querySelector('.aeo-modal-header');
            if (header && header.parentNode) {
                header.parentNode.insertBefore(err, body);
            }
        }
        return err;
    }
    function showInlineError(msg) {
        const err = ensureErrorContainer();
        if (!err) return;
        err.textContent = String(msg || 'Error');
        err.style.display = 'block';
    }
    function clearInlineError() {
        const err = modal ? modal.querySelector('#aeo-modal-error') : null;
        if (err) { err.textContent = ''; err.style.display = 'none'; }
    }

    function markFaqFieldError(input) {
        if (!input) return;
        input.setAttribute('aria-invalid', 'true');
        try { input.style.borderColor = '#d63638'; } catch (_) { /* noop */ }
        const parent = input.closest && input.closest('.aeo-faq-item');
        if (parent) { parent.classList.add('has-error'); }
    }

    function clearAllFaqErrors() {
        if (!faqsList) return;
        faqsList.querySelectorAll('.aeo-faq-item').forEach(function (item) {
            item.classList.remove('has-error');
            item.querySelectorAll('input,textarea').forEach(function (el) {
                el.removeAttribute('aria-invalid');
                if (el.style) { el.style.removeProperty('border-color'); }
            });
        });
    }

    function ensureFaqItemOpen(item) {
        if (!item) return;
        item.classList.add('is-open');
        const head = item.querySelector('.aeo-faq-head');
        if (head) { head.setAttribute('aria-expanded', 'true'); }
    }

    function validateFaqsForSave() {
        // Returns true when all FAQs are either fully filled (q+a) or there are no FAQs.
        // Blocks save if any item is empty or partial to avoid silent data loss.
        clearAllFaqErrors();
        let firstProblemField = null;
        let invalidCount = 0;
        if (!faqsList) return true;
        const items = faqsList.querySelectorAll('.aeo-faq-item');
        if (!items || items.length === 0) { return true; }
        items.forEach(function (item) {
            const q = item.querySelector('.aeo-faq-q');
            const a = item.querySelector('.aeo-faq-a');
            const qv = q ? (q.value || '').trim() : '';
            const av = a ? (a.value || '').trim() : '';
            const isComplete = (qv !== '' && av !== '');
            if (!isComplete) {
                invalidCount++;
                ensureFaqItemOpen(item);
                if (qv === '') { markFaqFieldError(q); if (!firstProblemField) firstProblemField = q; }
                if (av === '') { markFaqFieldError(a); if (!firstProblemField) firstProblemField = a; }
            }
        });
        if (invalidCount > 0) {
            showInlineError((window.wp && wp.i18n ? wp.i18n.__('Please complete or delete all FAQ items before saving.', 'aeo-content-plugin') : 'Please complete or delete all FAQ items before saving.'));
            try { if (firstProblemField && typeof firstProblemField.focus === 'function') { firstProblemField.focus(); } } catch (_) { /* noop */ }
            return false;
        }
        return true;
    }

    function showToast(msg, type) {
        try {
            const div = document.createElement('div');
            div.className = 'notice ' + (type === 'error' ? 'notice-error' : 'notice-success');
            div.style.position = 'fixed';
            div.style.right = '16px';
            div.style.bottom = '16px';
            div.style.zIndex = '100001';
            div.innerHTML = '<p>' + escapeHtml(msg || '') + '</p>';
            document.body.appendChild(div);
            setTimeout(function () { div.remove(); }, 3500);
        } catch (_) { /* noop */ }
    }

    function buildSaveParams() {
        const state = getCurrentState();
        const params = new URLSearchParams();
        params.append('action', 'aeo_editor_save');
        params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.editor ? window.aeoEditor.env.nonces.editor : '');
        params.append('post_id', String(window.aeoEditor.env.postId || 0));
        // Trim overview (server also validates)
        const ov = (state.overview || '').trim();
        params.append('overview', ov);
        // Encode FAQs as array indices, dropping empties
        let i = 0;
        (state.faqs || []).forEach(function (f) {
            const q = (f.question || '').trim();
            const a = (f.answer || '').trim();
            if (!q || !a) { return; }
            params.append('faqs[' + i + '][question]', q);
            params.append('faqs[' + i + '][answer]', a);
            i++;
        });
        return params;
    }

    function setSavingState(saving) {
        if (!saveBtn) return;
        if (saving) {
            saveBtn.setAttribute('disabled', 'disabled');
            saveBtn.textContent = (window.wp && wp.i18n ? wp.i18n.__('Saving…', 'aeo-content-plugin') : 'Saving…');
        } else {
            saveBtn.textContent = (window.wp && wp.i18n ? wp.i18n.__('Save', 'aeo-content-plugin') : 'Save');
            updateSaveButton();
        }
    }

    if (saveBtn) {
        saveBtn.addEventListener('click', function () {
            if (saveBtn.hasAttribute('disabled')) { return; }
            clearInlineError();
            if (!validateFaqsForSave()) {
                return;
            }
            setSavingState(true);
            const params = buildSaveParams();
            fetch(cfg.ajax_url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
                body: params.toString(),
                credentials: 'same-origin'
            }).then(function (r) { return r.json(); })
                .then(function (resp) {
                    if (!resp || !resp.success || !resp.data) {
                        const msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Save failed.';
                        throw new Error(msg);
                    }
                    const data = resp.data;
                    // Sync env and UI with sanitized server echo
                    window.aeoEditor.env.licenseActive = !!data.license_active;
                    window.aeoEditor.env.analyzed = !!data.analyzed;
                    if (overviewEl) { overviewEl.value = data.overview || ''; updateOverviewCount(); }
                    rebuildFaqsList(data.faqs || []);
                    snapshotState();
                    showToast((window.wp && wp.i18n ? wp.i18n.__('Saved.', 'aeo-content-plugin') : 'Saved.'), 'success');
                })
                .catch(function (err) {
                    const msg = (err && err.message) ? err.message : (window.wp && wp.i18n ? wp.i18n.__('Save failed.', 'aeo-content-plugin') : 'Save failed.');
                    showInlineError(msg);
                    showToast(msg, 'error');
                })
                .finally(function () {
                    setSavingState(false);
                });
        });
    }

    // ===== Phase 9: Analyze/Re-analyze Flow =====
    function setAnalyzingState(val) {
        isAnalyzing = !!val;
        if (val) {
            if (analyzeBtn) {
                analyzeBtn.setAttribute('disabled', 'disabled');
                analyzeBtn.textContent = (window.wp && wp.i18n ? wp.i18n.__('Analyzing…', 'aeo-content-plugin') : 'Analyzing…');
            }
            if (closeBtn) closeBtn.setAttribute('disabled', 'disabled');
            if (cancelBtn) cancelBtn.setAttribute('disabled', 'disabled');
            showOverlay((window.wp && wp.i18n ? wp.i18n.__('Analyzing…', 'aeo-content-plugin') : 'Analyzing…'));
            setFormEnabled(false);
        } else {
            updateAnalyzeLabel();
            setFormEnabled(true);
            if (closeBtn) closeBtn.removeAttribute('disabled');
            if (cancelBtn) cancelBtn.removeAttribute('disabled');
            hideOverlay();
        }
        updateSaveButton();
        updateAnalyzeButton();
    }

    function stopAnalysisPoll() {
        if (analysisPollTimer) {
            clearInterval(analysisPollTimer);
            analysisPollTimer = null;
        }
        isAnalysisPolling = false;
        analysisPollInFlight = false;
        analysisPollStartedAt = 0;
        setAnalyzingState(false);
    }

    function handleAnalyzeSuccess(data) {
        try { console.debug && console.debug('[AEO] analyze success; impr_count=', data.improvements_count, 'pending=', data.improvements_pending, 'pro_only=', data.improvements_pro_only); } catch (_) { }
        if (overviewEl) { overviewEl.value = data.overview || ''; updateOverviewCount(); }
        rebuildFaqsList(data.faqs || []);
        if (typeof data.license_active !== 'undefined') {
            window.aeoEditor.env.licenseActive = !!data.license_active;
        }
        if (data.trial) {
            setTrialState(data.trial);
        }
        window.aeoEditor.env.analyzed = !!data.analyzed;
        window.aeoEditor.env.autoStatus = 'done';
        updateAnalyzeLabel();
        snapshotState();
        if (Array.isArray(data.improvements)) {
            setImprovementsList(data.improvements);
        }
        // Populate Debug Log
        if (debugLogEl) {
            if (data.debug_log) {
                debugLogEl.value = JSON.stringify(data.debug_log, null, 2);
            } else {
                debugLogEl.value = 'No debug log available for this post.';
            }
        }
        // Refresh improvements after successful analyze, then decorate
        try { fetchImprovements().then(function () { try { decorateEditorSurface(); } catch (_) { } }); } catch (_) { /* noop */ }
        // Light upsell if PRO-only improvements are gated
        try {
            if (data && data.improvements_pro_only) {
                showToast((window.wp && wp.i18n ? wp.i18n.__('Free preview limit reached. Activate your license to keep generating improvements.', 'aeo-content-plugin') : 'Free preview limit reached. Activate your license to keep generating improvements.'), 'error');
            }
        } catch (_) { /* noop */ }
        showToast((window.wp && wp.i18n ? wp.i18n.__('Analysis updated.', 'aeo-content-plugin') : 'Analysis updated.'), 'success');
        // Notify other admin pages/tabs to refresh usage
        try { if ('BroadcastChannel' in window) { new BroadcastChannel('aeo_usage').postMessage({ type: 'refresh' }); } } catch (_) { }
        try { localStorage.setItem('aeo_usage_refresh', String(Date.now())); setTimeout(function () { try { localStorage.removeItem('aeo_usage_refresh'); } catch (_) { } }, 3000); } catch (_) { }
        // Refresh local gating from server after successful analyze
        try { refreshUsageStatus(); } catch (_) { }
    }

    function startAnalysisPoll(jobId) {
        if (!jobId || !cfg || !cfg.ajax_url) {
            stopAnalysisPoll();
            return;
        }
        if (analysisPollTimer) {
            clearInterval(analysisPollTimer);
        }
        isAnalysisPolling = true;
        analysisPollStartedAt = Date.now();
        const maxPollMs = 5 * 60 * 1000;
        const pollInterval = 4000;

        const poll = function () {
            if (!isAnalysisPolling) return;
            if (analysisPollInFlight) return;
            if ((Date.now() - analysisPollStartedAt) > maxPollMs) {
                stopAnalysisPoll();
                showInlineError((window.wp && wp.i18n ? wp.i18n.__('Analysis is taking too long. Please try again.', 'aeo-content-plugin') : 'Analysis is taking too long. Please try again.'));
                showToast((window.wp && wp.i18n ? wp.i18n.__('Analysis timed out.', 'aeo-content-plugin') : 'Analysis timed out.'), 'error');
                return;
            }
            analysisPollInFlight = true;
            const params = new URLSearchParams();
            params.append('action', 'aeo_poll_analysis_job');
            params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.analyze ? window.aeoEditor.env.nonces.analyze : '');
            params.append('post_id', String(window.aeoEditor.env.postId || 0));
            params.append('job_id', String(jobId));

            fetch(cfg.ajax_url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
                body: params.toString(),
                credentials: 'same-origin'
            }).then(function (r) { return r.json(); })
                .then(function (resp) {
                    analysisPollInFlight = false;
                    if (!isAnalysisPolling) { return; }
                    if (!resp || !resp.success) {
                        const data = (resp && resp.data) ? resp.data : {};
                        const msg = (data && data.message) ? data.message : (window.wp && wp.i18n ? wp.i18n.__('Analyze failed.', 'aeo-content-plugin') : 'Analyze failed.');
                        throw new Error(msg);
                    }
                    const data = resp.data || {};
                    if (data.pending) {
                        window.aeoEditor.env.autoStatus = data.status || 'running';
                        updateAnalyzeButton();
                        updateAnalyzeLabel();
                        return;
                    }
                    stopAnalysisPoll();
                    handleAnalyzeSuccess(data);
                })
                .catch(function (err) {
                    analysisPollInFlight = false;
                    if (!isAnalysisPolling) { return; }
                    const msg = (err && err.message) ? err.message : (window.wp && wp.i18n ? wp.i18n.__('Analyze failed.', 'aeo-content-plugin') : 'Analyze failed.');
                    stopAnalysisPoll();
                    showInlineError(msg);
                    showToast(msg, 'error');
                    try { refreshUsageStatus(); } catch (_) { }
                });
        };

        poll();
        analysisPollTimer = setInterval(poll, pollInterval);
    }

    if (analyzeBtn) {
        analyzeBtn.addEventListener('click', function () {
            if (analyzeBtn.hasAttribute('disabled')) return;
            clearInlineError();
            try { console.log('[AEO] Analyze clicked for post', window.aeoEditor && window.aeoEditor.env ? window.aeoEditor.env.postId : '(unknown)'); } catch (_) { }
            setAnalyzingState(true);
            const params = new URLSearchParams();
            params.append('action', 'aeo_analyze_post');
            params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.analyze ? window.aeoEditor.env.nonces.analyze : '');
            params.append('post_id', String(window.aeoEditor.env.postId || 0));

            const segments = segmentEditorContent();
            if (segments) {
                params.append('segments', JSON.stringify(segments));
            }

            fetch(cfg.ajax_url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
                body: params.toString(),
                credentials: 'same-origin'
            }).then(function (r) { return r.json(); })
                .then(function (resp) {
                    if (!resp || !resp.success) {
                        const data = (resp && resp.data) ? resp.data : {};
                        if (data && data.trial) {
                            setTrialState(data.trial);
                        }
                        const msg = (data && data.message) ? data.message : (window.wp && wp.i18n ? wp.i18n.__('Analyze failed.', 'aeo-content-plugin') : 'Analyze failed.');
                        throw new Error(msg);
                    }
                    if (!resp.data) {
                        throw new Error(window.wp && wp.i18n ? wp.i18n.__('Analyze failed.', 'aeo-content-plugin') : 'Analyze failed.');
                    }
                    const data = resp.data;
                    if (data.pending) {
                        if (typeof data.license_active !== 'undefined') {
                            window.aeoEditor.env.licenseActive = !!data.license_active;
                        }
                        if (data.trial) {
                            setTrialState(data.trial);
                        }
                        window.aeoEditor.env.autoStatus = data.status || 'queued';
                        updateAnalyzeButton();
                        updateAnalyzeLabel();
                        showToast(data.message || (window.wp && wp.i18n ? wp.i18n.__('Analysis queued.', 'aeo-content-plugin') : 'Analysis queued.'), 'success');
                        startAnalysisPoll(data.job_id);
                        return;
                    }
                    handleAnalyzeSuccess(data);
                })
                .catch(function (err) {
                    const msg = (err && err.message) ? err.message : (window.wp && wp.i18n ? wp.i18n.__('Analyze failed.', 'aeo-content-plugin') : 'Analyze failed.');
                    if (msg && /free run/i.test(msg)) {
                        setTrialState({ remaining: 0 });
                    }
                    showInlineError(msg);
                    showToast(msg, 'error');
                    // Also notify for a usage refresh (in case counters changed or limits were hit)
                    try { if ('BroadcastChannel' in window) { new BroadcastChannel('aeo_usage').postMessage({ type: 'refresh' }); } } catch (_) { }
                    try { localStorage.setItem('aeo_usage_refresh', String(Date.now())); setTimeout(function () { try { localStorage.removeItem('aeo_usage_refresh'); } catch (_) { } }, 3000); } catch (_) { }
                    // Fetch latest usage to update gating and banners
                    try { refreshUsageStatus(); } catch (_) { }
                    // If Pro quota likely exceeded, hint to open settings
                    try {
                        if (window.aeoEditor.env.licenseActive && /quota|limit|rate|429/i.test(String(msg))) {
                            const hint = (window.wp && wp.i18n) ? wp.i18n.__('Pro quota may be reached. See License Settings to manage usage.', 'aeo-content-plugin') : 'Pro quota may be reached. See License Settings to manage usage.';
                            showToast(hint, 'error');
                        }
                    } catch (_) { /* noop */ }
                })
                .finally(function () {
                    if (!isAnalysisPolling) {
                        setAnalyzingState(false);
                    }
                });
        });
    }

    // ===== Improvements: fetch + state (Phase 4, step 1) =====
    const impr = {
        items: [],
        mapById: new Map(),
    };

    function setImprovementsList(list) {
        const arr = Array.isArray(list) ? list.filter(function (item) { return item && typeof item === 'object'; }) : [];
        impr.items = arr;
        impr.mapById.clear();
        arr.forEach(function (it) {
            if (it && it.id) {
                impr.mapById.set(it.id, it);
            }
        });
        try { updateCountBadge(); } catch (_) { /* noop */ }
        try { kickDecorations(); } catch (_) { /* noop */ }
        return arr;
    }

    function fetchImprovements() {
        if (!hasPostId || !cfg || !cfg.ajax_url) return Promise.resolve([]);
        const params = new URLSearchParams();
        params.append('action', 'aeo_get_improvements');
        params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.editor ? window.aeoEditor.env.nonces.editor : '');
        params.append('post_id', String(window.aeoEditor.env.postId || 0));
        return fetch(cfg.ajax_url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
            body: params.toString(),
            credentials: 'same-origin'
        }).then(function (r) { return r.json(); })
            .then(function (resp) {
                if (!resp || !resp.success || !resp.data) return [];
                const list = Array.isArray(resp.data.improvements) ? resp.data.improvements : [];
                const normalized = setImprovementsList(list);
                try { console.debug && console.debug('[AEO] fetchImprovements count=', normalized.length, 'licenseActive=', window.aeoEditor.env.licenseActive); } catch (_) { }
                return normalized;
            }).catch(function () { return []; });
    }

    // ===== Improvements mapping + decorations (Gutenberg) =====
    function isGutenbergEnv() {
        if (hasClassicEditorDom()) { return false; }
        return !!(window.wp && wp.data && wp.data.select && window.aeoEditor.env.isGutenberg);
    }

    // Canvas (iframe) helpers for modern Gutenberg
    function getCanvasIframe() {
        try {
            return document.querySelector('iframe[aria-label="Editor canvas"], iframe.editor-canvas__iframe, .editor-canvas iframe, iframe[name="editor-canvas"]');
        } catch (_) { return null; }
    }
    function getCanvasDocument() {
        if (!isGutenbergEnv()) {
            return getClassicEditingDocument() || document;
        }
        try {
            const ifr = getCanvasIframe();
            if (ifr && ifr.contentDocument) return ifr.contentDocument;
            if (ifr && ifr.contentWindow && ifr.contentWindow.document) return ifr.contentWindow.document;
        } catch (_) { /* noop */ }
        return document; // fallback to same doc (older WP w/o iframe canvas)
    }

    function getClassicEditingDocument() {
        try {
            const editor = getClassicTinyMCEEditor();
            if (editor && typeof editor.getDoc === 'function') {
                const doc = editor.getDoc();
                if (doc) { return doc; }
            }
            if (editor && editor.iframeElement && editor.iframeElement.contentDocument) {
                return editor.iframeElement.contentDocument;
            }
            const textarea = getClassicTextarea();
            if (textarea && textarea.ownerDocument) {
                return textarea.ownerDocument;
            }
        } catch (_) { /* noop */ }
        return document;
    }
    function ensureCanvasStyles(doc) {
        try {
            if (!doc) return;
            if (doc.getElementById('aeo-canvas-style')) return;
            const style = doc.createElement('style');
            style.id = 'aeo-canvas-style';
            style.textContent = [
                '.aeo-impr-mark{position:relative;outline:2px dashed rgba(34,113,177,.35);outline-offset:2px;}',
                '.aeo-impr-veil{position:absolute;inset:0;z-index:2;background:rgba(255,255,255,.01);}',
                '.aeo-impr-badge{position:absolute;top:6px;right:6px;z-index:3;background:#2271b1;color:#fff;border-radius:10px;padding:2px 6px;font-size:11px;line-height:1.2;box-shadow:0 1px 2px rgba(0,0,0,.15)}',
                '.aeo-impr-badge.is-stale{background:#6c7781}',
                '.aeo-impr-badge::before{content:"✨";margin-right:4px}',
                '.aeo-impr-preview{position:absolute;top:32px;right:6px;z-index:4;width:min(440px,70vw);background:#fff;border:1px solid rgba(15,23,42,.08);border-radius:12px;box-shadow:0 18px 30px rgba(15,23,42,.18);padding:12px}',
                '.aeo-impr-preview .aeo-impr-preview-section{margin:0 0 10px 0}',
                '.aeo-impr-preview .aeo-impr-preview-section.is-secondary{opacity:.9}',
                '.aeo-impr-preview .aeo-impr-label{font-size:11px;color:#475569;margin-bottom:6px;font-weight:700;letter-spacing:.04em;text-transform:uppercase}',
                '.aeo-impr-preview .aeo-impr-text{font-size:12px;color:#0f172a;line-height:1.4}',
                '.aeo-impr-preview .aeo-impr-meta{display:flex;flex-wrap:wrap;gap:6px;margin-top:8px}',
                '.aeo-impr-preview .aeo-impr-pill{font-size:10px;line-height:1;border:1px solid #e2e8f0;color:#334155;background:#f8fafc;border-radius:999px;padding:4px 8px}',
                '.aeo-impr-preview .aeo-impr-pill.is-manual{background:#fff7ed;border-color:#fed7aa;color:#9a3412}',
                '.aeo-impr-preview .aeo-impr-text ul,.aeo-impr-preview .aeo-impr-text ol{margin:6px 0 6px 18px;padding:0}',
                '.aeo-impr-preview .aeo-impr-text table{width:100%;border-collapse:collapse;margin-top:6px;font-size:11px}',
                '.aeo-impr-preview .aeo-impr-text th,.aeo-impr-preview .aeo-impr-text td{border:1px solid #e2e8f0;padding:4px 6px;vertical-align:top}',
                '.aeo-impr-preview .aeo-impr-text thead th{background:#f8fafc}',
                '.aeo-impr-preview .aeo-impr-text caption{caption-side:top;text-align:left;font-weight:600;margin-bottom:4px}',
                '.aeo-impr-reason{font-size:11px;color:#475569;border-left:3px solid #e2e8f0;padding-left:8px;margin:10px 0 0 0}',
                '.aeo-impr-reason-label{font-weight:600;font-size:11px;margin:0 0 2px 0;}',
                '.aeo-impr-reason-body{font-size:11px;line-height:1.4;}',
                '.aeo-impr-actions{display:flex;flex-wrap:wrap;gap:8px;justify-content:flex-end;margin-top:12px}',
                /* Modern overlay buttons */
                '.aeo-btn{font-size:12px;line-height:1.1;font-weight:600;border:1px solid #d0d7de;background:#fff;color:#0f172a;padding:6px 14px;border-radius:999px;cursor:pointer;transition:transform .15s,box-shadow .2s,background .2s,border-color .2s;display:inline-flex;align-items:center;gap:6px;box-shadow:0 4px 10px rgba(15,23,42,.08);}',
                '.aeo-btn:hover{background:#f8fafc;border-color:#cbd5e1;transform:translateY(-1px);}',
                '.aeo-btn:active{transform:translateY(0);box-shadow:0 1px 2px rgba(15,23,42,.1) inset;}',
                '.aeo-btn:focus-visible{outline:2px solid rgba(34,113,177,.35);outline-offset:2px;}',
                '.aeo-btn-primary{background:linear-gradient(135deg,#2271b1,#0f4b88);border-color:#0f4b88;color:#fff;box-shadow:0 10px 18px rgba(34,113,177,.25);}',
                '.aeo-btn-primary:hover{background:linear-gradient(135deg,#1e63a0,#0d3f73);}',
                '.aeo-btn-primary:active{background:linear-gradient(135deg,#1a5a91,#0b365f);}',
                '.aeo-btn-secondary{background:#fff;color:#0f172a;}',
                '.aeo-btn-secondary:hover{background:#f1f5f9;}',
                '.aeo-btn[disabled],.aeo-btn[disabled]:hover{opacity:.55;cursor:not-allowed;background:#f8fafc;box-shadow:none;transform:none;}',
                '#aeo-impr-overlay-root{position:fixed;top:0;left:0;width:100%;height:100%;pointer-events:none;z-index:2147483646}',
                '.aeo-overlay-badge{pointer-events:auto;position:absolute;display:inline-flex;align-items:center;gap:6px;background:#111a22;color:#e3e8ef;border:1px solid #1f2d38;border-radius:24px;padding:4px 12px 4px 10px;font-size:12px;line-height:1;font-weight:600;box-shadow:0 4px 12px -2px rgba(0,0,0,.35),0 1px 2px rgba(0,0,0,.25);cursor:pointer;transition:background .35s,transform .25s,border-color .35s;}',
                '.aeo-overlay-badge:hover{background:#18242e;transform:translateY(-2px);}',
                '.aeo-overlay-badge:active{transform:translateY(0);}',
                '.aeo-overlay-badge.is-stale{background:#4d5560;color:#f8f9fa;border-color:#616b75;}',
                '.aeo-overlay-badge .aeo-badge-text{white-space:nowrap;}',
                /* Star icon styles for canvas injection */
                '.aeo-overlay-badge .aeo-star{position:relative;width:18px;height:18px;flex:0 0 18px;display:inline-block;}',
                '.aeo-overlay-badge .aeo-star::before{content:"✦";position:absolute;inset:0;display:flex;align-items:center;justify-content:center;font-size:20px;color:#4da3ff;text-shadow:0 0 3px rgba(77,163,255,0.5);}',
                '@supports (-webkit-background-clip: text){.aeo-overlay-badge .aeo-star::before{background:linear-gradient(130deg,#4da3ff,#7f5af0,#ff7edb,#ffa94d,#4da3ff);background-size:260% 260%;-webkit-background-clip:text;-webkit-text-fill-color:transparent;background-clip:text;animation:aeoStarGradient 5s linear infinite, aeoStarPulse 3.2s ease-in-out infinite;filter:drop-shadow(0 0 2px rgba(127,90,240,0.3));color:initial;text-shadow:none;}}',
                '.aeo-overlay-badge .aeo-star.is-stale::before{background:none;color:#cfd3d8;-webkit-text-fill-color:#cfd3d8;animation:none;filter:none;text-shadow:none;}',
                /* Keyframe animations for star effects */
                '@keyframes aeoStarGradient{0%{background-position:0% 40%;}50%{background-position:100% 60%;}100%{background-position:0% 40%;}}',
                '@keyframes aeoStarPulse{0%,100%{transform:scale(1);}50%{transform:scale(1.12);}}',
                '.aeo-overlay-box{--aeo-overlay-radius:4px;--aeo-overlay-stroke:2px;position:absolute;border-radius:var(--aeo-overlay-radius);pointer-events:none}',
                '.aeo-overlay-box::before{content:"";position:absolute;inset:calc(var(--aeo-overlay-stroke)*-1);border-radius:calc(var(--aeo-overlay-radius) + var(--aeo-overlay-stroke));padding:var(--aeo-overlay-stroke);background:linear-gradient(120deg,#4da3ff,#7f5af0,#ff7edb,#ffa94d,#4da3ff);background-size:300% 300%;animation:aeoOverlayStroke 6s linear infinite;opacity:.85;pointer-events:none;-webkit-mask:linear-gradient(#000 0 0) content-box,linear-gradient(#000 0 0);-webkit-mask-composite:xor;mask-composite:exclude;filter:drop-shadow(0 0 6px rgba(77,163,255,.35));}',
                '@keyframes aeoOverlayStroke{0%{background-position:0% 50%;}50%{background-position:100% 50%;}100%{background-position:0% 50%;}}',
                '@media (prefers-reduced-motion: reduce){.aeo-overlay-box::before{animation:none;}}',
                '.aeo-overlay-panel{pointer-events:none;position:absolute;background:#fff;border:1px solid rgba(15,23,42,.08);border-radius:12px;box-shadow:0 18px 30px rgba(15,23,42,.18);padding:12px;box-sizing:border-box;width:calc(100vw - 24px);max-width:560px;max-height:calc(100vh - 24px);overflow:auto;overscroll-behavior:contain;-webkit-overflow-scrolling:touch;z-index:2147483647;opacity:0;transform:translateY(var(--aeo-shift,4px));transition:opacity .18s ease,transform .18s ease;}',
                '.aeo-overlay-panel[data-position="above"]{--aeo-shift:-4px;}',
                '.aeo-overlay-panel.is-active{opacity:1;transform:translateY(0);pointer-events:auto;}',
                '.aeo-overlay-panel .aeo-impr-preview{position:static;top:auto;right:auto;width:auto;background:transparent;border:none;box-shadow:none;padding:0;}',
            ].join('');
            doc.head && doc.head.appendChild(style);
        } catch (_) { /* noop */ }
    }

    // Overlay helpers (render UI outside editable content)
    const overlayRegistries = new WeakMap();

    let overlayDebugFlag = null;
    function computeOverlayDebugDefault() {
        let flag = false;
        try {
            if (typeof window !== 'undefined') {
                if (window.aeoOverlayDebug === true) {
                    flag = true;
                } else if (window.localStorage && window.localStorage.getItem('aeoOverlayDebug') === '1') {
                    flag = true;
                }
            }
        } catch (_) { /* noop */ }
        return flag;
    }

    function isOverlayDebugEnabled() {
        if (overlayDebugFlag === null) {
            overlayDebugFlag = computeOverlayDebugDefault();
        }
        return !!overlayDebugFlag;
    }

    function overlayLog() {
        if (!isOverlayDebugEnabled()) { return; }
        try {
            const args = Array.prototype.slice.call(arguments);
            args.unshift('[AEO overlay]');
            console.debug.apply(console, args);
        } catch (_) { /* noop */ }
    }

    if (typeof window !== 'undefined') {
        window.aeoOverlaySetDebug = function (enabled) {
            const next = (typeof enabled === 'undefined') ? !isOverlayDebugEnabled() : !!enabled;
            overlayDebugFlag = next;
            try {
                if (window.localStorage) {
                    window.localStorage.setItem('aeoOverlayDebug', next ? '1' : '0');
                }
            } catch (_) { /* noop */ }
            overlayLog('debug toggled', next);
            return next;
        };
    }

    function getOverlayRegistry(doc) {
        if (!doc) { return null; }
        let reg = overlayRegistries.get(doc);
        if (!reg) {
            reg = {
                doc: doc,
                root: null,
                entries: [],
                resizeObserver: null,
                mutationObserver: null,
                scrollContainers: new Map(),
                onScroll: null,
                onResize: null,
                raf: null
            };
            overlayRegistries.set(doc, reg);
        }
        return reg;
    }

    function teardownOverlayRegistry(reg) {
        if (!reg) { return; }
        const win = (reg.doc && reg.doc.defaultView) || window;
        if (reg.resizeObserver) {
            try { reg.resizeObserver.disconnect(); } catch (_) { /* noop */ }
        }
        if (reg.mutationObserver) {
            try { reg.mutationObserver.disconnect(); } catch (_) { /* noop */ }
        }
        reg.resizeObserver = null;
        reg.mutationObserver = null;
        if (reg.scrollContainers && reg.scrollContainers.size) {
            reg.scrollContainers.forEach(function (handler, target) {
                try { target.removeEventListener('scroll', handler); } catch (_) { /* noop */ }
            });
            reg.scrollContainers.clear();
        }
        if (win && win.removeEventListener) {
            if (reg.onScroll) { win.removeEventListener('scroll', reg.onScroll); }
            if (reg.onResize) { win.removeEventListener('resize', reg.onResize); }
        }
        reg.onScroll = null;
        reg.onResize = null;
        if (reg.raf) {
            cancelAnimationFrame(reg.raf);
            reg.raf = null;
        }
        reg.entries = [];
    }

    function scheduleOverlaySync(reg) {
        if (!reg) { return; }
        if (reg.raf) { cancelAnimationFrame(reg.raf); }
        overlayLog('scheduleOverlaySync', { entries: reg.entries ? reg.entries.length : 0 });
        reg.raf = requestAnimationFrame(function () {
            reg.raf = null;
            syncOverlayEntries(reg);
        });
    }

    function syncOverlayEntries(reg) {
        if (!reg || !reg.entries || !reg.entries.length) { return; }
        overlayLog('syncOverlayEntries', { count: reg.entries.length });
        for (let i = 0; i < reg.entries.length; i++) {
            positionOverlayElements(reg.entries[i]);
        }
    }

    function findScrollableAncestors(node, doc) {
        const list = [];
        if (!node || !doc || !doc.defaultView) { return list; }
        const win = doc.defaultView;
        const seen = new Set();
        let el = node.parentElement;
        while (el && el !== doc.body && el !== doc.documentElement) {
            try {
                const style = win.getComputedStyle(el);
                const overflowY = style ? style.overflowY : '';
                const overflowX = style ? style.overflowX : '';
                const combined = (overflowY || '') + (overflowX || '');
                if (/(auto|scroll|overlay)/i.test(combined)) {
                    if (!seen.has(el)) {
                        seen.add(el);
                        list.push(el);
                    }
                }
            } catch (_) { /* noop */ }
            el = el.parentElement;
        }
        return list;
    }

    function setOverlayEntries(doc, entries) {
        const reg = getOverlayRegistry(doc);
        if (!reg) { return; }
        const win = (doc && doc.defaultView) || window;
        overlayLog('setOverlayEntries', {
            entryCount: Array.isArray(entries) ? entries.length : 0,
            docContext: (doc && doc !== document) ? 'iframe' : 'top-document'
        });

        if (reg.resizeObserver) {
            try { reg.resizeObserver.disconnect(); } catch (_) { /* noop */ }
        }
        if (typeof ResizeObserver !== 'undefined') {
            if (!reg.resizeObserver) {
                reg.resizeObserver = new ResizeObserver(function () {
                    scheduleOverlaySync(reg);
                });
            }
        } else {
            reg.resizeObserver = null;
        }

        reg.entries = Array.isArray(entries) ? entries : [];

        if (reg.resizeObserver) {
            reg.entries.forEach(function (entry) {
                if (entry && entry.node) {
                    try { reg.resizeObserver.observe(entry.node); } catch (_) { /* noop */ }
                }
            });
            try {
                if (doc && doc.documentElement) { reg.resizeObserver.observe(doc.documentElement); }
            } catch (_) { /* noop */ }
            try {
                if (doc && doc.body) { reg.resizeObserver.observe(doc.body); }
            } catch (_) { /* noop */ }
        }

        if (typeof MutationObserver !== 'undefined' && doc) {
            if (!reg.mutationObserver) {
                reg.mutationObserver = new MutationObserver(function () {
                    scheduleOverlaySync(reg);
                });
            }
            try {
                reg.mutationObserver.disconnect();
                if (doc.body) {
                    reg.mutationObserver.observe(doc.body, { attributes: true, attributeFilter: ['class', 'style'] });
                }
                if (doc.documentElement) {
                    reg.mutationObserver.observe(doc.documentElement, { attributes: true, attributeFilter: ['class', 'style'] });
                }
            } catch (_) { /* noop */ }
        }

        if (!reg.scrollContainers) {
            reg.scrollContainers = new Map();
        }
        const desiredScrollTargets = new Set();
        reg.entries.forEach(function (entry) {
            findScrollableAncestors(entry && entry.node, doc).forEach(function (el) {
                if (el && el.addEventListener) {
                    desiredScrollTargets.add(el);
                }
            });
        });

        const existingTargets = Array.from(reg.scrollContainers.keys());
        existingTargets.forEach(function (target) {
            if (!desiredScrollTargets.has(target)) {
                const handler = reg.scrollContainers.get(target);
                try { target.removeEventListener('scroll', handler); } catch (_) { /* noop */ }
                reg.scrollContainers.delete(target);
                overlayLog('scroll container removed', {
                    tag: target && target.tagName,
                    className: target && target.className
                });
            }
        });

        desiredScrollTargets.forEach(function (target) {
            if (reg.scrollContainers.has(target)) { return; }
            const handler = function () {
                overlayLog('scroll container event', {
                    tag: target && target.tagName,
                    className: target && target.className
                });
                scheduleOverlaySync(reg);
            };
            try {
                target.addEventListener('scroll', handler, { passive: true });
            } catch (_) {
                try { target.addEventListener('scroll', handler); } catch (err2) { /* noop */ }
            }
            reg.scrollContainers.set(target, handler);
            overlayLog('scroll container added', {
                tag: target && target.tagName,
                className: target && target.className
            });
        });

        overlayLog('scroll container summary', {
            total: reg.scrollContainers ? reg.scrollContainers.size : 0
        });

        if (win && win.addEventListener) {
            if (!reg.onScroll) {
                reg.onScroll = function () { scheduleOverlaySync(reg); };
                win.addEventListener('scroll', reg.onScroll, { passive: true });
            }
            if (!reg.onResize) {
                reg.onResize = function () { scheduleOverlaySync(reg); };
                win.addEventListener('resize', reg.onResize);
            }
        }

        scheduleOverlaySync(reg);
    }

    function repositionOverlayPanel(entry) {
        if (!entry || !entry.panel || !entry.badge || !entry.doc) { return; }
        const panel = entry.panel;
        const badge = entry.badge;
        const doc = entry.doc;
        const win = (doc && doc.defaultView) || window;

        const viewportWidth = win.innerWidth || 1280;
        const viewportHeight = win.innerHeight || 720;
        const spacing = 12;

        const wasHidden = panel.style.display === 'none';
        const prevVisibility = panel.style.visibility;
        if (wasHidden) {
            panel.style.visibility = 'hidden';
            panel.style.display = 'block';
        }
        const panelRect = panel.getBoundingClientRect();
        if (wasHidden) {
            panel.style.display = 'none';
            panel.style.visibility = prevVisibility || '';
        }

        const badgeRect = badge.getBoundingClientRect();
        let top = badgeRect.bottom + spacing;
        let placement = 'below';
        if ((viewportHeight - badgeRect.bottom) < (panelRect.height + spacing) && badgeRect.top > (panelRect.height + spacing)) {
            top = badgeRect.top - panelRect.height - spacing;
            placement = 'above';
        }
        if (top < spacing) { top = spacing; }
        if (top + panelRect.height > viewportHeight - spacing) {
            top = Math.max(spacing, viewportHeight - panelRect.height - spacing);
        }

        let left = badgeRect.right - panelRect.width;
        if (left < spacing) { left = spacing; }
        if (left + panelRect.width > viewportWidth - spacing) {
            left = Math.max(spacing, viewportWidth - panelRect.width - spacing);
        }

        panel.dataset.position = placement;
        panel.style.left = Math.round(left) + 'px';
        panel.style.top = Math.round(top) + 'px';
        overlayLog('repositionOverlayPanel', {
            placement: placement,
            left: panel.style.left,
            top: panel.style.top,
            viewportWidth: viewportWidth,
            viewportHeight: viewportHeight
        });
    }

    function positionOverlayElements(entry) {
        if (!entry || !entry.node || !entry.box || !entry.doc) { return; }
        const rect = entry.node.getBoundingClientRect();
        const hasSize = rect && rect.width > 0 && rect.height > 0;

        if (!hasSize) {
            overlayLog('positionOverlayElements: hidden (no size)', {
                nodeTag: entry.node && entry.node.tagName,
                rect: rect
            });
            entry.box.style.display = 'none';
            if (entry.badge) { entry.badge.style.display = 'none'; }
            if (entry.panel && !entry.panel.classList.contains('is-active')) {
                entry.panel.style.display = 'none';
            }
            return;
        }

        entry.box.style.display = 'block';
        entry.box.style.left = Math.round(rect.left - 4) + 'px';
        entry.box.style.top = Math.round(rect.top - 4) + 'px';
        entry.box.style.width = Math.round(rect.width + 8) + 'px';
        entry.box.style.height = Math.round(rect.height + 8) + 'px';
        overlayLog('positionOverlayElements: box', {
            nodeTag: entry.node && entry.node.tagName,
            left: entry.box.style.left,
            top: entry.box.style.top,
            width: entry.box.style.width,
            height: entry.box.style.height
        });

        if (entry.badge) {
            entry.badge.style.display = 'inline-flex';
            const badgeWidth = entry.badge.getBoundingClientRect().width || 0;
            const badgeLeft = Math.round(rect.right - badgeWidth);
            entry.badge.style.left = Math.max(8, badgeLeft) + 'px';
            entry.badge.style.top = Math.round(rect.top - 18) + 'px';
            overlayLog('positionOverlayElements: badge', {
                left: entry.badge.style.left,
                top: entry.badge.style.top,
                badgeWidth: badgeWidth
            });
        }

        if (entry.panel && entry.panel.classList.contains('is-active')) {
            repositionOverlayPanel(entry);
        }
    }

    function ensureOverlayRoot(doc) {
        try {
            if (!doc) return null;
            let root = doc.getElementById('aeo-impr-overlay-root');
            if (!root) {
                root = doc.createElement('div');
                root.id = 'aeo-impr-overlay-root';
                doc.body && doc.body.appendChild(root);
                overlayLog('ensureOverlayRoot: created', {
                    docContext: (doc && doc !== document) ? 'iframe' : 'top-document'
                });
            } else {
                overlayLog('ensureOverlayRoot: existing', {
                    docContext: (doc && doc !== document) ? 'iframe' : 'top-document'
                });
            }
            const reg = getOverlayRegistry(doc);
            if (reg) { reg.root = root; }
            return root;
        } catch (_) { return null; }
    }
    function clearOverlay(doc) {
        try {
            const root = doc && doc.getElementById && doc.getElementById('aeo-impr-overlay-root');
            if (root) {
                root.innerHTML = '';
                overlayLog('clearOverlay: cleared root', {
                    docContext: (doc && doc !== document) ? 'iframe' : 'top-document'
                });
            }
            const reg = getOverlayRegistry(doc);
            if (reg) { teardownOverlayRegistry(reg); }
        } catch (_) { /* noop */ }
    }

    function getSupportedBlocks() {
        if (!isGutenbergEnv()) return [];
        try {
            const sel = wp.data.select('core/block-editor');
            const blocks = sel.getBlocks();
            const flat = [];
            (function walk(list) {
                (list || []).forEach(function (b) {
                    flat.push(b);
                    if (b.innerBlocks && b.innerBlocks.length) walk(b.innerBlocks);
                });
            })(blocks);
            return flat.filter(function (b) {
                return b && (
                    b.name === 'core/paragraph' ||
                    b.name === 'core/heading' ||
                    b.name === 'core/list-item' ||
                    b.name === 'core/pullquote'
                );
            });
        } catch (_) { return []; }
    }

    // Strip any accidentally persisted AEO helper nodes from block attributes.
    function sanitizeParagraphContentAttributes() {
        if (!isGutenbergEnv()) return;
        try {
            const sel = wp.data.select('core/block-editor');
            const dispatch = wp.data.dispatch('core/block-editor');
            const blocks = sel.getBlocks();

            // Helper to sanitize a single block
            const sanitizeBlock = function (b) {
                if (!b) return;
                const isSupported = (
                    b.name === 'core/paragraph' ||
                    b.name === 'core/heading' ||
                    b.name === 'core/list-item' ||
                    b.name === 'core/pullquote'
                );

                if (isSupported) {
                    let html = '';
                    let attrName = 'content';
                    if (b.attributes && typeof b.attributes.content === 'string') {
                        html = b.attributes.content;
                    } else if (b.attributes && typeof b.attributes.value === 'string') {
                        html = b.attributes.value;
                        attrName = 'value';
                    }

                    if (html && html.indexOf('aeo-impr-') !== -1) {
                        try {
                            const temp = document.createElement('div');
                            temp.innerHTML = html;
                            // Remove our helper nodes from within the content
                            temp.querySelectorAll('.aeo-impr-badge, .aeo-impr-veil, .aeo-impr-preview').forEach(function (n) { n.remove(); });
                            const clean = temp.innerHTML;
                            if (clean !== html) {
                                const update = {};
                                update[attrName] = clean;
                                dispatch.updateBlockAttributes(b.clientId, update);
                            }
                        } catch (_) { /* noop per block */ }
                    }
                }

                if (b.innerBlocks && b.innerBlocks.length) {
                    b.innerBlocks.forEach(sanitizeBlock);
                }
            };

            (blocks || []).forEach(sanitizeBlock);
        } catch (_) { /* noop */ }
    }

    function tryFindParagraphNodeByContent(docCtx, normText) {
        const docs = [docCtx || document, document];
        for (let d = 0; d < docs.length; d++) {
            const doc = docs[d];
            if (!doc || !doc.querySelectorAll) continue;
            const nodes = doc.querySelectorAll('.wp-block-paragraph, p');
            for (let i = 0; i < nodes.length; i++) {
                const n = nodes[i];
                const t = normalizeTextForHash(n.textContent || '');
                if (!t) continue;
                if (t === normText) return n;
                if (textSimilarityDice(t, normText) >= 0.98) return n;
            }
        }
        return null;
    }

    function normalizeTextForHash(s) {
        if (s == null) return '';
        let decoded = '';
        try {
            const tmp = document.createElement('div');
            tmp.innerHTML = String(s);
            decoded = tmp.textContent || tmp.innerText || '';
        } catch (_) {
            decoded = String(s);
        }
        // Aggressively clean the string to match PHP
        // 1. Remove the BOM character and other zero-width spaces
        // 2. Normalize all whitespace (including non-breaking spaces) to a single space
        // 3. Trim whitespace from start/end
        return decoded
            .replace(/[\u200B-\u200D\uFEFF]/g, '') // Remove zero-width spaces and BOM
            .replace(/<[^>]*>/g, ' ')             // Redundant but safe: strip tags
            .replace(/\s+/g, ' ')                 // Collapse all whitespace
            .trim();
    }

    // Canonicalize text for human-equivalent comparison (tolerant to punctuation/nbsp differences)
    function normalizeForComparison(s) {
        if (s == null) return '';
        let t = '';
        try {
            const tmp = document.createElement('div');
            tmp.innerHTML = String(s);
            t = tmp.textContent || tmp.innerText || '';
        } catch (_) {
            t = String(s);
        }
        try { t = t.normalize('NFKC'); } catch (_) { /* older browsers */ }
        // Replace NBSP and various Unicode spaces with regular spaces
        t = t.replace(/[\u00A0\u2000-\u200B\u202F\u205F\u3000]/g, ' ');
        // Normalize smart quotes/apostrophes to straight
        t = t.replace(/[\u2018\u2019\u2032]/g, "'").replace(/[\u201C\u201D\u2033]/g, '"');
        // Normalize dashes/minus to hyphen
        t = t.replace(/[\u2013\u2014\u2212]/g, '-');
        // Ellipsis to three dots
        t = t.replace(/\u2026/g, '...');
        // Collapse whitespace
        t = t.replace(/\s+/g, ' ').trim();
        return t;
    }

    // Minimal SHA-1 implementation (hex) for deterministic stale checks (fallback when Web Crypto unavailable)
    function sha1Sync(str) {
        function rotl(n, s) { return (n << s) | (n >>> (32 - s)); }
        function toHex(n) { let h = (n >>> 0).toString(16); return '00000000'.slice(h.length) + h; }
        // UTF-8 encode
        str = unescape(encodeURIComponent(str));
        const msg = [];
        for (let i = 0; i < str.length; i++) msg.push(str.charCodeAt(i));
        const l = msg.length;
        msg.push(0x80);
        while ((msg.length % 64) !== 56) msg.push(0);
        const bitLen = l * 8;
        for (let i = 7; i >= 0; i--) msg.push((bitLen >>> (i * 8)) & 0xff);
        let H0 = 0x67452301, H1 = 0xEFCDAB89, H2 = 0x98BADCFE, H3 = 0x10325476, H4 = 0xC3D2E1F0;
        const w = new Array(80);
        for (let i = 0; i < msg.length; i += 64) {
            for (let j = 0; j < 16; j++) {
                const k = i + j * 4;
                w[j] = (msg[k] << 24) | (msg[k + 1] << 16) | (msg[k + 2] << 8) | (msg[k + 3]);
            }
            for (let j = 16; j < 80; j++) w[j] = rotl(w[j - 3] ^ w[j - 8] ^ w[j - 14] ^ w[j - 16], 1);
            let a = H0, b = H1, c = H2, d = H3, e = H4;
            for (let j = 0; j < 80; j++) {
                let f, k;
                if (j < 20) { f = (b & c) | (~b & d); k = 0x5A827999; }
                else if (j < 40) { f = b ^ c ^ d; k = 0x6ED9EBA1; }
                else if (j < 60) { f = (b & c) | (b & d) | (c & d); k = 0x8F1BBCDC; }
                else { f = b ^ c ^ d; k = 0xCA62C1D6; }
                const temp = (rotl(a, 5) + f + e + k + (w[j] >>> 0)) >>> 0;
                e = d; d = c; c = rotl(b, 30) >>> 0; b = a; a = temp;
            }
            H0 = (H0 + a) >>> 0;
            H1 = (H1 + b) >>> 0;
            H2 = (H2 + c) >>> 0;
            H3 = (H3 + d) >>> 0;
            H4 = (H4 + e) >>> 0;
        }
        return toHex(H0) + toHex(H1) + toHex(H2) + toHex(H3) + toHex(H4);
    }

    async function sha1Hex(str) {
        try {
            if (window.crypto && crypto.subtle && window.isSecureContext) {
                const enc = new TextEncoder();
                const buf = await crypto.subtle.digest('SHA-1', enc.encode(str));
                const view = new DataView(buf);
                let hex = '';
                for (let i = 0; i < view.byteLength; i++) {
                    const b = view.getUint8(i).toString(16).padStart(2, '0');
                    hex += b;
                }
                return hex;
            }
        } catch (_) { /* fall back */ }
        return Promise.resolve(sha1Sync(str));
    }

    // Similarity (Dice coefficient on bigrams) for resilient paragraph matching
    function textSimilarityDice(a, b) {
        a = normalizeTextForHash(a || '');
        b = normalizeTextForHash(b || '');
        if (!a || !b) return 0;
        function bigrams(s) { const out = []; for (let i = 0; i < s.length - 1; i++) { out.push(s.slice(i, i + 2)); } return out; }
        const A = bigrams(a), B = bigrams(b);
        if (!A.length || !B.length) return 0;
        const freq = new Map();
        for (let i = 0; i < A.length; i++) { const k = A[i]; freq.set(k, (freq.get(k) || 0) + 1); }
        let overlap = 0;
        for (let i = 0; i < B.length; i++) { const k = B[i]; const v = freq.get(k) || 0; if (v > 0) { overlap++; freq.set(k, v - 1); } }
        return (2 * overlap) / (A.length + B.length);
    }

    // Similarity but assumes inputs are already normalized for comparison
    function textSimilarityDiceNormalized(a, b) {
        if (!a || !b) return 0;
        function bigrams(s) { const out = []; for (let i = 0; i < s.length - 1; i++) { out.push(s.slice(i, i + 2)); } return out; }
        const A = bigrams(a), B = bigrams(b);
        if (!A.length || !B.length) return 0;
        const freq = new Map();
        for (let i = 0; i < A.length; i++) { const k = A[i]; freq.set(k, (freq.get(k) || 0) + 1); }
        let overlap = 0;
        for (let i = 0; i < B.length; i++) { const k = B[i]; const v = freq.get(k) || 0; if (v > 0) { overlap++; freq.set(k, v - 1); } }
        return (2 * overlap) / (A.length + B.length);
    }

    // Remove all Gutenberg decorations
    function clearDecorations() {
        try {
            const docs = [document, getCanvasDocument()];
            docs.forEach(function (d) {
                if (!d) return;
                d.querySelectorAll('.aeo-impr-preview').forEach(function (el) { el.remove(); });
                d.querySelectorAll('.aeo-impr-veil').forEach(function (el) { el.remove(); });
                d.querySelectorAll('.aeo-impr-badge').forEach(function (el) { el.remove(); });
                d.querySelectorAll('.aeo-impr-mark').forEach(function (el) { el.classList.remove('aeo-impr-mark'); });
                clearOverlay(d);
            });
        } catch (_) { /* noop */ }
    }

    // Build the preview card DOM for an improvement
    function makePreviewCard(item, opts, docCtx) {
        const stale = !!(opts && opts.stale);
        const doc = docCtx || document;
        const wrap = doc.createElement('div');
        wrap.className = 'aeo-impr-preview';
        wrap.setAttribute('role', 'dialog');
        wrap.setAttribute('aria-label', (window.wp && wp.i18n ? wp.i18n.__('Content improvement', 'aeo-content-plugin') : 'Content improvement'));

        const titleSuggest = (window.wp && wp.i18n ? wp.i18n.__('Suggested rewrite', 'aeo-content-plugin') : 'Suggested rewrite');
        const titleOriginal = (window.wp && wp.i18n ? wp.i18n.__('Original', 'aeo-content-plugin') : 'Original');
        const applyLabel = (window.wp && wp.i18n ? wp.i18n.__('Apply', 'aeo-content-plugin') : 'Apply');
        const markLabel = (window.wp && wp.i18n ? wp.i18n.__('Mark applied', 'aeo-content-plugin') : 'Mark applied');
        const copyLabel = (window.wp && wp.i18n ? wp.i18n.__('Copy HTML', 'aeo-content-plugin') : 'Copy HTML');
        const rejectLabel = (window.wp && wp.i18n ? wp.i18n.__('Reject', 'aeo-content-plugin') : 'Reject');
        const staleMsg = (window.wp && wp.i18n ? wp.i18n.__('Suggestion is stale. Re-run Analyze to refresh.', 'aeo-content-plugin') : 'Suggestion is stale. Re-run Analyze to refresh.');

        const suggestedHtml = (item && typeof item.suggestedHtml === 'string') ? item.suggestedHtml : '';
        const suggested = suggestedHtml ? suggestedHtml : escapeHtml(item && item.suggestedText ? item.suggestedText : '');
        const original = escapeHtml(item && item.originalText ? item.originalText : '');
        const reason = escapeHtml(item && item.reason ? item.reason : '');
        const score = (item && typeof item.score === 'number') ? ('(' + Math.round(item.score * 100) + '%)') : '';
        const structureType = normalizeStructureType(item && item.structureType);
        const manual = isManualImprovement(item);
        const structureLabel = (structureType !== 'paragraph') ? formatStructureTypeLabel(structureType) : '';
        const metaBits = [];
        if (structureLabel) {
            metaBits.push('<span class="aeo-impr-pill">' + (window.wp && wp.i18n ? wp.i18n.__('Structured', 'aeo-content-plugin') : 'Structured') + ': ' + structureLabel + '</span>');
        }
        if (manual) {
            metaBits.push('<span class="aeo-impr-pill is-manual">' + (window.wp && wp.i18n ? wp.i18n.__('Manual apply', 'aeo-content-plugin') : 'Manual apply') + '</span>');
        }
        const metaHtml = metaBits.length ? ('<div class="aeo-impr-meta">' + metaBits.join('') + '</div>') : '';
        const showCopy = !!(suggestedHtml || (item && item.suggestedText));
        const applyText = manual ? markLabel : applyLabel;

        wrap.innerHTML = [
            '<div class="aeo-impr-preview-inner">',
            '  <div class="aeo-impr-preview-section">',
            '    <div class="aeo-impr-label">' + titleSuggest + ' ' + score + '</div>',
            '    <div class="aeo-impr-text aeo-impr-text-suggested">' + suggested + '</div>',
            metaHtml,
            '  </div>',
            '  <div class="aeo-impr-preview-section is-secondary">',
            '    <div class="aeo-impr-label">' + titleOriginal + '</div>',
            '    <div class="aeo-impr-text aeo-impr-text-original">' + original + '</div>',
            '  </div>',
            (function () {
                if (!reason) { return ''; }
                var reasonLabel = (window.wp && wp.i18n ? wp.i18n.__('Reason', 'aeo-content-plugin') : 'Reason');
                return '  <div class="aeo-impr-reason">' +
                    '<div class="aeo-impr-reason-label">' + reasonLabel + '</div>' +
                    '<div class="aeo-impr-reason-body">' + reason + '</div>' +
                    '</div>';
            })(),
            '  <div class="aeo-impr-actions">',
            (showCopy ? ('    <button type="button" class="aeo-btn aeo-btn-secondary aeo-impr-copy">' + copyLabel + '</button>') : ''),
            '    <button type="button" class="aeo-btn aeo-btn-primary aeo-impr-accept"' + (stale ? ' disabled title="' + staleMsg + '"' : '') + '>' + applyText + '</button>',
            '    <button type="button" class="aeo-btn aeo-btn-secondary aeo-impr-reject">' + rejectLabel + '</button>',
            '  </div>',
            '</div>'
        ].join('');
        return wrap;
    }

    // RAF-batched decorations trigger
    let decoIv = null;
    function kickDecorations() {
        if (decoIv) { cancelAnimationFrame(decoIv); }
        decoIv = requestAnimationFrame(function () { decorateEditorSurface(); });
    }

    function decorateEditorSurface() {
        if (!window.aeoEditor.env.licenseActive) {
            clearDecorations();
            toggleClassicTextHint(false);
            try { console.debug && console.debug('[AEO] decorate: skipped – license inactive'); } catch (_) { }
            return;
        }
        if (!impr.items || !impr.items.length) {
            clearDecorations();
            toggleClassicTextHint(false);
            try { console.debug && console.debug('[AEO] decorate: no improvements'); } catch (_) { }
            return;
        }
        if (isGutenbergEnv()) {
            toggleClassicTextHint(false);
            decorateGutenbergBlocks();
        } else {
            if (!areClassicOverlaysEnabled()) {
                clearDecorations();
                toggleClassicTextHint(false);
                return;
            }
            decorateClassicEditor();
        }
    }

    // Gutenberg decorations and interactions
    function decorateGutenbergBlocks() {
        if (!isGutenbergEnv()) return;
        const iframeDoc = getCanvasDocument();
        ensureCanvasStyles(iframeDoc);
        // Only clear overlay; do not touch block DOM
        clearOverlay(iframeDoc);
        const paraBlocks = getSupportedBlocks();
        if (!paraBlocks.length) { try { console.debug && console.debug('[AEO] decorateBlocks: no supported blocks'); } catch (_) { } return; }
        overlayLog('decorateBlocks:start', {
            improvements: impr.items.length,
            blocks: paraBlocks.length,
            docContext: (iframeDoc && iframeDoc !== document) ? 'iframe' : 'top-document'
        });



        const candidates = [];
        let validIndex = 0;

        paraBlocks.forEach(function (b) {
            let wrapper = null;
            if (iframeDoc && iframeDoc.querySelector) {
                wrapper = iframeDoc.querySelector(`[data-block="${b.clientId}"]`);
            }
            if (!wrapper && document && document.querySelector) {
                wrapper = document.querySelector(`[data-block="${b.clientId}"]`);
            }

            // Don't just rely on b.attributes.content. It can be unreliable.
            // Get text directly from the rendered DOM node for maximum accuracy.
            let liveText = '';
            let attrName = 'content';

            if (wrapper && wrapper.textContent) {
                liveText = wrapper.textContent;
            } else if (b.attributes) {
                if (typeof b.attributes.content === 'string') {
                    liveText = b.attributes.content;
                } else if (typeof b.attributes.value === 'string') {
                    liveText = b.attributes.value;
                    attrName = 'value';
                }
            }

            // Determine attribute name for updates if not already set by fallback
            if (b.attributes && typeof b.attributes.value === 'string' && typeof b.attributes.content === 'undefined') {
                attrName = 'value';
            }

            const norm = normalizeTextForHash(liveText);

            // Backend skips empty text, so we must too to keep index in sync
            if (!norm) { return; }

            const nodeResolved = wrapper || null;

            candidates.push({
                clientId: b.clientId,
                // index: validIndex++, // Index removed
                text: norm,
                hash: sha1Sync(norm),
                blockName: b.name,
                node: nodeResolved,
                attrContent: liveText,
                attrName: attrName
            });
        });

        const byHash = new Map();
        candidates.forEach(function (c) { if (c.hash) byHash.set(String(c.hash), c); });

        try { console.debug && console.debug('[AEO] decorateBlocks: candidates=', candidates.length, 'items=', impr.items.length); } catch (_) { }

        const overlayRoot = ensureOverlayRoot(iframeDoc);
        if (!overlayRoot) return;

        const overlayEntries = [];

        impr.items.forEach(function (item, i) {
            if (!item || !item.locator) return;
            let match = null;
            const targetHash = (item.locator && item.locator.hash) ? String(item.locator.hash) : null;
            if (targetHash && byHash.has(targetHash)) {
                match = byHash.get(targetHash);
            }
            if (!match && item.locator.clientId) {
                match = candidates.find(function (c) { return c.clientId === item.locator.clientId; });
            }
            // Index matching removed
            // if (!match && typeof item.locator.index === 'number') {
            //    match = candidates.find(function (c) { return c.index === item.locator.index; });
            // }
            if (!match) {
                const origNorm = normalizeTextForHash(item.originalText || '');
                let best = null, bestScore = 0;
                for (let i = 0; i < candidates.length; i++) {
                    const c = candidates[i];
                    const s = textSimilarityDice(c.text, origNorm);
                    if (s > bestScore) { bestScore = s; best = c; }
                }
                if (best && bestScore >= 0.62) { match = best; }
            }
            if (!match) {
                overlayLog('decorateBlocks:item-no-match', {
                    index: i,
                    hash: targetHash,
                    locator: item.locator
                });
                try { console.debug && console.debug('[AEO] item', i, 'no match; hash=', targetHash, 'loc=', item.locator); } catch (_) { }
                return;
            }

            if (!match.node) {
                try { console.debug && console.debug('[AEO] item', i, 'matched but no node'); } catch (_) { }
                return;
            }

            const el = match.node; // wrapper or editable element
            const curNorm = normalizeTextForHash(match.attrContent || match.text || '');
            const curHash = sha1Sync(curNorm);
            // Robust comparison tolerant to typography differences
            const curComp = normalizeForComparison(match.attrContent || match.text || '');
            const origComp = normalizeForComparison(item.originalText || '');
            const sim = textSimilarityDiceNormalized(curComp, origComp);
            const sameEnough = sim >= 0.98; // tolerate minor smart quotes/nbsp/dash differences
            let isStale;
            if (targetHash) {
                // Hash mismatch alone does not imply stale if text is effectively the same
                isStale = (curHash !== targetHash) && !sameEnough;
            } else {
                isStale = !sameEnough;
            }
            overlayLog('decorateBlocks:item-match', {
                index: i,
                stale: !!isStale,
                clientId: match.clientId,
                hashMismatch: targetHash ? (curHash !== targetHash) : null,
                similarity: sim
            });
            try { console.debug && console.debug('[AEO] item matched at idx=', match.index, 'stale=', !!isStale); } catch (_) { }
            const outline = iframeDoc.createElement('div');
            outline.className = 'aeo-overlay-box';
            overlayRoot.appendChild(outline);

            const badge = iframeDoc.createElement('div');
            badge.className = 'aeo-overlay-badge' + (isStale ? ' is-stale' : '');
            badge.title = isStale
                ? (window.wp && wp.i18n ? wp.i18n.__('Suggestion is stale. Re-run Analyze.', 'aeo-content-plugin') : 'Suggestion is stale. Re-run Analyze.')
                : (window.wp && wp.i18n ? wp.i18n.__('View suggestion', 'aeo-content-plugin') : 'View suggestion');
            var badgeLabel = (window.wp && wp.i18n ? wp.i18n.__('Improve', 'aeo-content-plugin') : 'Improve');
            badge.innerHTML = '<span class="aeo-star" aria-hidden="true"></span><span class="aeo-badge-text">' + badgeLabel + '</span>';
            overlayRoot.appendChild(badge);

            const panel = iframeDoc.createElement('div');
            panel.className = 'aeo-overlay-panel';
            panel.style.display = 'none';
            const preview = makePreviewCard(item, { stale: isStale }, iframeDoc);
            const accept = preview.querySelector('.aeo-impr-accept');
            const reject = preview.querySelector('.aeo-impr-reject');
            const copy = preview.querySelector('.aeo-impr-copy');
            if (accept) accept.addEventListener('click', function (e) {
                e.preventDefault();
                if (accept.hasAttribute('disabled')) return;
                if (isManualImprovement(item)) { markImprovementApplied(item); return; }
                applyImprovement(item, match);
            });
            if (reject) reject.addEventListener('click', function (e) { e.preventDefault(); rejectImprovement(item); });
            if (copy) copy.addEventListener('click', function (e) { e.preventDefault(); copySuggestedHtml(item, copy); });
            panel.appendChild(preview);
            overlayRoot.appendChild(panel);

            const overlayEntry = {
                node: el,
                box: outline,
                badge: badge,
                panel: panel,
                doc: iframeDoc
            };
            overlayEntries.push(overlayEntry);
            try { positionOverlayElements(overlayEntry); } catch (_) { /* noop */ }

            function showPanel() {
                positionOverlayElements(overlayEntry);
                if (panel._hideTimer) { clearTimeout(panel._hideTimer); panel._hideTimer = null; }
                panel.style.display = 'block';
                repositionOverlayPanel(overlayEntry);
                requestAnimationFrame(function () { panel.classList.add('is-active'); });
            }
            function hidePanelSoon() {
                if (panel._hideTimer) clearTimeout(panel._hideTimer);
                panel._hideTimer = setTimeout(function () {
                    panel._hideTimer = null;
                    const overBadge = badge && badge.matches ? badge.matches(':hover') : false;
                    const overPanel = panel && panel.matches ? panel.matches(':hover') : false;
                    if (overBadge || overPanel) { return; }
                    panel.classList.remove('is-active');
                    const onEnd = function (e) {
                        if (e && e.target !== panel) return;
                        panel.removeEventListener('transitionend', onEnd);
                        if (!panel.classList.contains('is-active')) { panel.style.display = 'none'; }
                    };
                    panel.addEventListener('transitionend', onEnd, { once: true });
                }, 200);
            }

            badge.addEventListener('mouseenter', showPanel);
            badge.addEventListener('mouseleave', hidePanelSoon);
            panel.addEventListener('mouseenter', showPanel);
            panel.addEventListener('mouseleave', hidePanelSoon);
        });

        setOverlayEntries(iframeDoc, overlayEntries);
    }

    function decorateClassicEditor() {
        const mode = detectClassicEditorMode();
        if (mode !== 'classic-visual') {
            toggleClassicTextHint(true);
            clearDecorations();
            return;
        }
        toggleClassicTextHint(false);

        const editor = getClassicTinyMCEEditor();
        const doc = getClassicEditingDocument();
        if (!editor || !doc || !doc.body) {
            clearDecorations();
            return;
        }

        ensureCanvasStyles(doc);
        clearOverlay(doc);

        const candidates = getClassicParagraphCandidates(doc);
        if (!candidates.length) {
            setOverlayEntries(doc, []);
            return;
        }

        overlayLog('decorateClassicEditor:start', {
            improvements: impr.items.length,
            paragraphs: candidates.length
        });

        const byHash = new Map();
        candidates.forEach(function (c) {
            if (c.hash) { byHash.set(String(c.hash), c); }
        });

        const overlayRoot = ensureOverlayRoot(doc);
        if (!overlayRoot) { return; }

        const overlayEntries = [];

        impr.items.forEach(function (item, i) {
            if (!item || !item.locator) { return; }
            const targetHash = item.locator && item.locator.hash ? String(item.locator.hash) : null;
            let match = targetHash && byHash.get(targetHash) ? byHash.get(targetHash) : null;

            // Index matching removed per user request
            // if (!match && typeof item.locator.index === 'number') {
            //    match = candidates[item.locator.index] || null;
            // }

            if (!match) {
                const origNorm = normalizeTextForHash(item.originalText || '');
                let best = null; let bestScore = 0;
                for (let j = 0; j < candidates.length; j++) {
                    const c = candidates[j];
                    const score = textSimilarityDice(c.text, origNorm);
                    if (score > bestScore) {
                        bestScore = score;
                        best = c;
                    }
                }
                if (best && bestScore >= 0.62) {
                    match = best;
                }
            }

            if (!match) {
                overlayLog('decorateClassicEditor:item-no-match', {
                    index: i,
                    hash: targetHash
                });
                return;
            }

            const liveText = match.attrContent || match.text || '';
            const curNorm = normalizeTextForHash(liveText);
            const curHash = sha1Sync(curNorm);
            const curComp = normalizeForComparison(liveText);
            const origComp = normalizeForComparison(item.originalText || '');
            const sim = textSimilarityDiceNormalized(curComp, origComp);
            const sameEnough = sim >= 0.98;
            const isStale = targetHash ? ((curHash !== targetHash) && !sameEnough) : !sameEnough;

            const outline = doc.createElement('div');
            outline.className = 'aeo-overlay-box';
            overlayRoot.appendChild(outline);

            const badge = doc.createElement('div');
            badge.className = 'aeo-overlay-badge' + (isStale ? ' is-stale' : '');
            badge.title = isStale
                ? (window.wp && wp.i18n ? wp.i18n.__('Suggestion is stale. Re-run Analyze.', 'aeo-content-plugin') : 'Suggestion is stale. Re-run Analyze.')
                : (window.wp && wp.i18n ? wp.i18n.__('View suggestion', 'aeo-content-plugin') : 'View suggestion');
            var badgeLabel = (window.wp && wp.i18n ? wp.i18n.__('Improve', 'aeo-content-plugin') : 'Improve');
            badge.innerHTML = '<span class="aeo-star" aria-hidden="true"></span><span class="aeo-badge-text">' + badgeLabel + '</span>';
            overlayRoot.appendChild(badge);

            const panel = doc.createElement('div');
            panel.className = 'aeo-overlay-panel';
            panel.style.display = 'none';
            const preview = makePreviewCard(item, { stale: isStale }, doc);
            const accept = preview.querySelector('.aeo-impr-accept');
            const reject = preview.querySelector('.aeo-impr-reject');
            const copy = preview.querySelector('.aeo-impr-copy');
            if (accept) accept.addEventListener('click', function (e) {
                e.preventDefault();
                if (accept.hasAttribute('disabled')) return;
                if (isManualImprovement(item)) { markImprovementApplied(item); return; }
                applyClassicImprovement(item, match);
            });
            if (reject) reject.addEventListener('click', function (e) { e.preventDefault(); rejectImprovement(item); });
            if (copy) copy.addEventListener('click', function (e) { e.preventDefault(); copySuggestedHtml(item, copy); });
            panel.appendChild(preview);
            overlayRoot.appendChild(panel);

            const overlayEntry = {
                node: match.node,
                box: outline,
                badge: badge,
                panel: panel,
                doc: doc
            };
            overlayEntries.push(overlayEntry);
            try { positionOverlayElements(overlayEntry); } catch (_) { /* noop */ }

            function showPanel() {
                positionOverlayElements(overlayEntry);
                if (panel._hideTimer) { clearTimeout(panel._hideTimer); panel._hideTimer = null; }
                panel.style.display = 'block';
                repositionOverlayPanel(overlayEntry);
                requestAnimationFrame(function () { panel.classList.add('is-active'); });
            }
            function hidePanelSoon() {
                if (panel._hideTimer) clearTimeout(panel._hideTimer);
                panel._hideTimer = setTimeout(function () {
                    panel._hideTimer = null;
                    const overBadge = badge && badge.matches ? badge.matches(':hover') : false;
                    const overPanel = panel && panel.matches ? panel.matches(':hover') : false;
                    if (overBadge || overPanel) { return; }
                    panel.classList.remove('is-active');
                    const onEnd = function (e) {
                        if (e && e.target !== panel) return;
                        panel.removeEventListener('transitionend', onEnd);
                        if (!panel.classList.contains('is-active')) { panel.style.display = 'none'; }
                    };
                    panel.addEventListener('transitionend', onEnd, { once: true });
                }, 200);
            }

            badge.addEventListener('mouseenter', showPanel);
            badge.addEventListener('mouseleave', hidePanelSoon);
            panel.addEventListener('mouseenter', showPanel);
            panel.addEventListener('mouseleave', hidePanelSoon);
        });

        setOverlayEntries(doc, overlayEntries);
    }

    function getClassicParagraphCandidates(doc) {
        if (!doc || !doc.body) { return []; }
        // Removed blockquote to avoid duplication with inner paragraphs, matching backend logic
        const selectors = 'p, li, h1, h2, h3, h4, h5, h6';
        const nodes = doc.body.querySelectorAll(selectors);
        const items = [];
        // let index = 0; // Indexing removed

        // Debug capture
        const debugList = [];

        nodes.forEach(function (node) {
            // Safety: ensure blockquote is not double-counted if querySelectorAll behaves unexpectedly
            if (node.tagName === 'BLOCKQUOTE') { return; }

            // Filter out internal TinyMCE UI elements and shortcode previews
            if (node.hasAttribute('data-mce-bogus')) { return; }
            if (node.classList.contains('mce-offscreen')) { return; }
            // We used to filter .wpview, but that causes valid content (like Blockquotes in Classic) to be skipped if they are wrapped in views.
            // Since backend expands shortcodes/blocks, we should index the frontend view too.
            // if (node.closest && node.closest('.wpview, .mce-preview-object')) { return; }

            const rawText = node.textContent || node.innerText || '';
            const text = normalizeTextForHash(rawText);

            // Filter out empty or shortcode-only paragraphs (e.g. [gallery])
            if (!text) { return; }
            if (/^\[.+\]$/.test(text.trim())) { return; }

            debugList.push({ tag: node.tagName, text: text.substring(0, 15) });

            items.push({
                node: node,
                // index: index++,
                text: text,
                hash: sha1Sync(text),
                attrContent: rawText
            });
        });

        // Expose for debug tab
        if (window.aeoEditor) { window.aeoEditor.lastCandidates = debugList; }

        if (!items.length) {
            const fallback = doc.body.textContent || '';
            const norm = normalizeTextForHash(fallback);
            if (norm) {
                items.push({
                    node: doc.body,
                    index: 0,
                    text: norm,
                    hash: sha1Sync(norm),
                    attrContent: fallback
                });
            }
        }
        return items;
    }

    function resolveClassicCandidateForItem(item, fallbackMatch, doc) {
        if (!item || !doc) { return null; }
        if (fallbackMatch && fallbackMatch.node && doc.contains(fallbackMatch.node)) {
            return fallbackMatch;
        }
        const candidates = getClassicParagraphCandidates(doc);
        if (!candidates.length) { return null; }
        const locator = item.locator || {};
        const targetHash = locator && locator.hash ? String(locator.hash) : null;
        if (targetHash) {
            const byHash = candidates.find(function (c) { return c.hash === targetHash; });
            if (byHash) { return byHash; }
        }
        // Index matching removed
        // if (typeof locator.index === 'number' && locator.index >= 0 && locator.index < candidates.length) {
        //    return candidates[locator.index];
        // }
        const origNorm = normalizeTextForHash(item.originalText || '');
        if (!origNorm) { return candidates[0]; }
        let best = null;
        let bestScore = 0;
        candidates.forEach(function (c) {
            const score = textSimilarityDice(c.text, origNorm);
            if (score > bestScore) {
                bestScore = score;
                best = c;
            }
        });
        if (best && bestScore >= 0.6) { return best; }
        return null;
    }

    function applyImprovement(item, match) {
        if (!isGutenbergEnv()) return;
        if (isManualImprovement(item)) { markImprovementApplied(item); return; }
        if (!match || !match.clientId) { return; }
        const suggested = getSuggestedMarkup(item);
        if (!suggested) { return; }
        let applied = false;
        try {
            const dispatch = wp.data.dispatch('core/block-editor');
            const canReplace = suggestionNeedsBlockReplace(suggested) && match.blockName !== 'core/list-item';
            if (canReplace) {
                const blocks = parseSuggestedBlocks(suggested);
                if (blocks && blocks.length) {
                    dispatch.replaceBlocks(match.clientId, blocks);
                    applied = true;
                }
            }
            if (!applied) {
                const update = {};
                const targetTag = resolveTargetTagFromMatch(match);
                update[match.attrName || 'content'] = deriveSuggestedInnerHtml(targetTag, suggested);
                dispatch.updateBlockAttributes(match.clientId, update);
                applied = true;
            }
        } catch (_) { /* noop */ }
        if (!applied) { return; }
        resolveImprovement(item.id, 'apply').then(function () {
            fetchImprovements().then(function () { decorateEditorSurface(); });
            showToast((window.wp && wp.i18n ? wp.i18n.__('Applied.', 'aeo-content-plugin') : 'Applied.'), 'success');
        });
    }

    function applyClassicImprovement(item, match) {
        if (!item) { return; }
        if (isManualImprovement(item)) { markImprovementApplied(item); return; }
        if (!areClassicOverlaysEnabled()) {
            const msg = (window.wp && wp.i18n
                ? wp.i18n.__('Classic suggestions are temporarily disabled by an administrator.', 'aeo-content-plugin')
                : 'Classic suggestions are temporarily disabled by an administrator.');
            showToast(msg, 'info');
            return;
        }
        const mode = detectClassicEditorMode();
        if (mode !== 'classic-visual') {
            const msg = (window.wp && wp.i18n
                ? wp.i18n.__('Switch back to the Visual tab to apply suggestions.', 'aeo-content-plugin')
                : 'Switch back to the Visual tab to apply suggestions.');
            showToast(msg, 'info');
            return;
        }
        const editor = getClassicTinyMCEEditor();
        const doc = getClassicEditingDocument();
        if (!editor || !doc) {
            const err = (window.wp && wp.i18n
                ? wp.i18n.__('Classic editor is not ready yet. Try again in a moment.', 'aeo-content-plugin')
                : 'Classic editor is not ready yet. Try again in a moment.');
            showToast(err, 'error');
            return;
        }

        const target = resolveClassicCandidateForItem(item, match, doc);
        if (!target || !target.node) {
            const err = (window.wp && wp.i18n
                ? wp.i18n.__('Could not find the paragraph to update. Re-run Analyze.', 'aeo-content-plugin')
                : 'Could not find the paragraph to update. Re-run Analyze.');
            showToast(err, 'error');
            return;
        }

        const suggested = getSuggestedMarkup(item);
        if (!suggested) { return; }
        const replaceBlock = suggestionNeedsBlockReplace(suggested);
        const innerHtml = replaceBlock ? '' : deriveSuggestedInnerHtml(target.node.tagName, suggested);

        try {
            editor.undoManager.transact(function () {
                if (replaceBlock) {
                    try {
                        if (editor.dom && typeof editor.dom.setOuterHTML === 'function') {
                            editor.dom.setOuterHTML(target.node, suggested);
                        } else if (typeof target.node.outerHTML === 'string') {
                            target.node.outerHTML = String(suggested);
                        } else {
                            target.node.innerHTML = String(suggested);
                        }
                    } catch (_) {
                        target.node.innerHTML = String(suggested);
                    }
                } else {
                    try {
                        target.node.innerHTML = innerHtml;
                    } catch (_) {
                        target.node.textContent = innerHtml;
                    }
                }
            });
            editor.nodeChanged();
            editor.fire('change');
        } catch (err) {
            const msg = (window.wp && wp.i18n
                ? wp.i18n.__('Update failed. Try reloading the editor.', 'aeo-content-plugin')
                : 'Update failed. Try reloading the editor.');
            showToast(msg, 'error');
            return;
        }

        resolveImprovement(item.id, 'apply').then(function () {
            fetchImprovements().then(function () { decorateEditorSurface(); });
            showToast((window.wp && wp.i18n ? wp.i18n.__('Applied.', 'aeo-content-plugin') : 'Applied.'), 'success');
        });
    }

    function rejectImprovement(item) {
        resolveImprovement(item.id, 'reject').then(function () {
            fetchImprovements().then(function () { decorateEditorSurface(); });
            showToast((window.wp && wp.i18n ? wp.i18n.__('Rejected.', 'aeo-content-plugin') : 'Rejected.'), 'success');
        });
    }

    function resolveImprovement(id, action) {
        if (!id) return Promise.resolve();
        const params = new URLSearchParams();
        params.append('action', action === 'apply' ? 'aeo_apply_improvement' : 'aeo_reject_improvement');
        params.append('nonce', window.aeoEditor.env.nonces && window.aeoEditor.env.nonces.editor ? window.aeoEditor.env.nonces.editor : '');
        params.append('post_id', String(window.aeoEditor.env.postId || 0));
        params.append('id', id);
        return fetch(cfg.ajax_url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
            body: params.toString(),
            credentials: 'same-origin'
        }).then(function (r) { return r.json(); }).then(function (resp) { return resp; }).catch(function () { return null; });
    }

    // Hook count badge updates after functions are defined (run once)
    (function installImprovementsBadgePatches() {
        if (installImprovementsBadgePatches._done) return;
        try {
            if (typeof fetchImprovements === 'function' && typeof resolveImprovement === 'function') {
                const _origFetchImprovements = fetchImprovements;
                fetchImprovements = function () {
                    return _origFetchImprovements();
                };
                const _origResolveImprovement = resolveImprovement;
                resolveImprovement = function (id, action) {
                    return _origResolveImprovement(id, action).then(function (resp) {
                        return fetchImprovements().then(function () { return resp; });
                    });
                };
                installImprovementsBadgePatches._done = true;
            }
        } catch (_) { /* noop */ }
    })();

    // Kick an initial improvements fetch + sanitize + decorate on editor load
    try {
        if (hasPostId && cfg && cfg.ajax_url) {
            sanitizeParagraphContentAttributes();
            fetchImprovements().then(function () { try { decorateEditorSurface(); } catch (_) { } });
        }
    } catch (_) { /* noop */ }

    // Subscribe to Gutenberg block changes to keep decorations in sync
    (function subscribeToBlockChanges() {
        if (!isGutenbergEnv()) return;
        try {
            const sel = wp.data.select('core/block-editor');
            let lastSig = '';
            const mkSig = function () {
                try {
                    const blocks = sel.getBlocks();
                    // signature using id+name+content to avoid heavy objects
                    return JSON.stringify(blocks.map(function (b) { return [b.clientId, b.name, (b.attributes && b.attributes.content) || '']; }));
                } catch (_) { return ''; }
            };
            lastSig = mkSig();
            let raf = null;
            let sanitizedOnce = false;
            wp.data.subscribe(function () {
                const cur = mkSig();
                if (cur !== lastSig) {
                    lastSig = cur;
                    if (raf) cancelAnimationFrame(raf);
                    raf = requestAnimationFrame(function () {
                        try {
                            if (!sanitizedOnce) { sanitizeParagraphContentAttributes(); sanitizedOnce = true; }
                            decorateGutenbergBlocks();
                        } catch (_) { }
                    });
                }
            });
            // Re-run when the canvas iframe reloads
            function attachIframeListeners() {
                const ifr = getCanvasIframe();
                if (!ifr) return false;
                if (ifr._aeoAttached) return true;
                ifr._aeoAttached = true;

                ifr.addEventListener('load', function () { try { sanitizeParagraphContentAttributes(); decorateGutenbergBlocks(); } catch (_) { } }, { once: false });
                // Reposition overlays on scroll/resize inside iframe
                try {
                    const idoc = ifr.contentDocument || ifr.contentWindow && ifr.contentWindow.document;
                    const win = ifr.contentWindow || (idoc && idoc.defaultView);
                    if (win) {
                        let raf2 = null;
                        const rerender = function () {
                            if (raf2) cancelAnimationFrame(raf2);
                            raf2 = requestAnimationFrame(function () { try { decorateGutenbergBlocks(); } catch (_) { } });
                        };
                        win.addEventListener('scroll', rerender, { passive: true });
                        win.addEventListener('resize', rerender, { passive: true });
                    }
                } catch (_) { /* noop */ }

                // Force a decoration pass now that we have the iframe
                requestAnimationFrame(function () { try { decorateGutenbergBlocks(); } catch (_) { } });
                return true;
            }

            if (!attachIframeListeners()) {
                let tries = 0;
                const iv = setInterval(function () {
                    tries++;
                    if (attachIframeListeners() || tries >= 50) { // 10s
                        clearInterval(iv);
                    }
                }, 200);
            }
        } catch (_) { /* best effort */ }
    })();
})();
