(
    function ($) {
        'use strict';

        // Bulk poll interval in milliseconds
        var BULK_POLL_INTERVAL = 2000; // Consider 2000ms (2 seconds) for balanced server load and user experience

        // Function to show a toast-like notification
        function showAeoNotice(message, isError) {
            var noticeHtml = '<div class="aeo-notice ' + (isError ? 'error' : 'updated') + '"><p>' + message + '</p></div>';
            $(noticeHtml)
                .appendTo('body')
                .fadeIn()
                .delay(4000)
                .fadeOut(function () {
                    $(this).remove();
                });
        }

        $(document).ready(function () {
            var $spinner = $('#aeo-license-spinner');
            // Cross-tab/page usage refresh broadcast
            var usageChannel = null;
            try { if ('BroadcastChannel' in window) { usageChannel = new BroadcastChannel('aeo_usage'); } } catch (e) { usageChannel = null; }
            function notifyUsageRefresh() {
                try { if (usageChannel) { usageChannel.postMessage({ type: 'refresh' }); } } catch (_) { /* noop */ }
                try {
                    localStorage.setItem('aeo_usage_refresh', String(Date.now()));
                    setTimeout(function () { try { localStorage.removeItem('aeo_usage_refresh'); } catch (_) { } }, 3000);
                } catch (_) { /* noop */ }
                try { if ($('#aeo-usage-status').length && typeof fetchUsage === 'function') { fetchUsage(true); } } catch (_) { /* noop */ }
            }
            // Receive notifications (only relevant on settings page where usage UI exists)
            try { if (usageChannel) { usageChannel.onmessage = function (ev) { if (ev && ev.data && ev.data.type === 'refresh') { if ($('#aeo-usage-status').length) { fetchUsage(true); } } }; } } catch (_) { /* noop */ }
            window.addEventListener('storage', function (e) {
                if (e && e.key === 'aeo_usage_refresh' && $('#aeo-usage-status').length) { fetchUsage(true); }
            });
            // Usage UI elements
            var $usageWrap = $('#aeo-usage-status');
            var $planPill = $('#aeo-plan-pill');
            var $usageLine = $('#aeo-usage-line');
            var $usageBar = $usageWrap.find('.aeo-usage-bar');
            var $usageFill = $usageWrap.find('.aeo-usage-fill');
            var $usageFallback = $('#aeo-usage-fallback');
            var $upgradeActions = $('.aeo-upgrade-actions');
            var $subscriptionNote = $('.aeo-subscription-note');
            var siteTz = (aeo_ajax_object && aeo_ajax_object.timezone) ? aeo_ajax_object.timezone : '';
            var lastUsageAt = 0; // simple client cache timestamp

            function cachePlan(plan) {
                try {
                    localStorage.setItem('aeo_plan_cache', JSON.stringify({ plan: plan, ts: Date.now() }));
                } catch (e) { /* noop */ }
            }
            function toggleUpgradeUi(plan) {
                if ($upgradeActions.length) {
                    if (plan === 'pro') {
                        $upgradeActions.attr('hidden', 'hidden');
                    } else {
                        $upgradeActions.removeAttr('hidden');
                    }
                }
                if ($subscriptionNote.length) {
                    if (plan === 'pro') {
                        $subscriptionNote.removeAttr('hidden');
                    } else {
                        $subscriptionNote.attr('hidden', 'hidden');
                    }
                }
            }

            function fmtResetDate(iso) {
                if (!iso) return '';
                try {
                    var date = new Date(iso);
                    var opts = { month: 'short', day: 'numeric' };
                    if (siteTz) opts.timeZone = siteTz;
                    return new Intl.DateTimeFormat(undefined, opts).format(date);
                } catch (e) { return ''; }
            }

            function renderUsage(data) {
                if (!data || typeof data !== 'object') { showFallback(); return; }
                var plan = (data.plan === 'pro') ? 'pro' : 'free';
                cachePlan(plan);
                toggleUpgradeUi(plan);
                // Plan pill
                $planPill.text(plan === 'pro' ? 'Pro' : 'Free')
                    .removeClass('is-pro is-free')
                    .addClass(plan === 'pro' ? 'is-pro' : 'is-free');

                // Line + bar
                var label = '';
                var percent = 0;
                if (plan === 'pro' && data.pro) {
                    var max = Number(data.pro.max || 300);
                    var used = Number(data.pro.used_in_window || 0);
                    var remaining = Math.max(0, max - used);
                    var resetAt = data.pro.reset_at || null;
                    var resetStr = resetAt ? fmtResetDate(resetAt) : '';
                    label = 'Pro • ' + remaining + ' remaining' + (resetStr ? ' (resets ' + resetStr + ')' : '');
                    percent = max > 0 ? Math.min(100, Math.round((used / max) * 100)) : 0;
                } else if (data.free) {
                    var limit = Number(data.free.limit || 10);
                    var usedF = Number(data.free.used || 0);
                    var remainF = Math.max(0, limit - usedF);
                    label = usedF + '/' + limit + ' used (' + remainF + ' remaining)';
                    percent = limit > 0 ? Math.min(100, Math.round((usedF / limit) * 100)) : 0;
                } else {
                    showFallback();
                    return;
                }
                $usageLine.text(label);
                $usageBar.attr('aria-valuenow', String(percent));
                $usageFill.css('width', percent + '%');
                $usageBar.show();
                $usageFallback.attr('hidden', true);
            }

            function showFallback() {
                $usageBar.hide();
                $usageFallback.removeAttr('hidden').text('Unable to fetch usage');
            }

            function fetchUsage(force) {
                var now = Date.now();
                if (!force && lastUsageAt && (now - lastUsageAt) < 15000) { return; }
                $.post(ajaxurl, {
                    action: 'aeo_usage_status',
                    nonce: aeo_ajax_object.status_nonce
                }, function (resp) {
                    if (resp && resp.success && resp.data) {
                        lastUsageAt = now;
                        renderUsage(resp.data);
                    } else {
                        showFallback();
                    }
                }).fail(function () { showFallback(); });
            }

            // Initial fetch on the license page only (element exists)
            if ($usageWrap.length) {
                fetchUsage(true);
            }
            var bulkActive = ($('td.column-aeo_analysis .aeo-bulk-status[data-status="queued"], td.column-aeo_analysis .aeo-bulk-status[data-status="running"]').length > 0);
            var bulkNonce = aeo_ajax_object && aeo_ajax_object.bulk_nonce ? aeo_ajax_object.bulk_nonce : '';

            function pollBulk() {
                if (!bulkActive) return;
                var ids = [];
                $('td.column-aeo_analysis .aeo-bulk-status').each(function () {
                    ids.push($(this).data('post-id'));
                });
                if (!ids.length) { setTimeout(pollBulk, BULK_POLL_INTERVAL); return; }
                $.post(ajaxurl, { action: 'aeo_bulk_poll', ids: ids, nonce: bulkNonce }, function (resp) {
                    if (!resp || !resp.success) { setTimeout(pollBulk, BULK_POLL_INTERVAL); return; }
                    var d = resp.data;
                    // Progress bar removed by design; no top-level progress updates.
                    // Update per-row statuses
                    Object.keys(d.statuses).forEach(function (pid) {
                        var st = d.statuses[pid];
                        var $wrap = $('.aeo-bulk-status[data-post-id="' + pid + '"]');
                        if (!$wrap.length) return;
                        var map = {
                            'queued': 'Queued', 'running': 'Running…', 'done': 'Done', 'failed': 'Failed', 'skipped': 'Skipped', 'canceled': 'Canceled'
                        };
                        var spinner = (st === 'queued' || st === 'running');
                        $wrap.attr('data-status', st).html((spinner ? '<span class="spinner is-active" style="float:none;margin:0 4px 0 0;"></span>' : '') + '<span class="aeo-bulk-status-text">' + (map[st] || '') + '</span>');
                    });
                    if (!d.active) {
                        bulkActive = false;
                        $('#posts-filter').find('input, select, button').prop('disabled', false);
                        $('.aeo-analyze-button, .aeo-reanalyze-button').prop('disabled', false).removeClass('disabled');
                        setTimeout(function () {
                            location.reload();
                        }, 1000);
                        return;
                    }
                    setTimeout(pollBulk, BULK_POLL_INTERVAL);
                }).fail(function () { setTimeout(pollBulk, 8000); });
            }
            if (bulkActive) {
                // Disable manual analyze buttons globally
                $('.aeo-analyze-button, .aeo-reanalyze-button').prop('disabled', true).addClass('disabled');
                // Cancel button removed from UI for simplified UX; users should wait for bulk actions to complete or reload the page.
            }

            // Cancel button removed from UI; no handler.

            // Handle Activation
            $(document).on('click', '#aeo-activate-license', function (e) {
                e.preventDefault();
                var $button = $(this);
                var licenseKey = $('#aeo_license_key_input').val();

                if (!licenseKey) {
                    alert('Please enter a license key.');
                    return;
                }

                $spinner.addClass('is-active');
                $button.prop('disabled', true);

                $.ajax({
                    url: aeo_ajax_object.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aeo_activate_license',
                        nonce: aeo_ajax_object.license_nonce,
                        license_key: licenseKey
                    },
                    success: function (response) {
                        if (response && response.success) {
                            alert(response.data && response.data.message ? response.data.message : 'License activated.');
                            // Refresh usage immediately; then reload (to flip UI controls)
                            fetchUsage(true);
                            notifyUsageRefresh();
                            setTimeout(function () { location.reload(); }, 300);
                        } else {
                            var msg = response && response.data && response.data.message ? response.data.message : 'Activation failed.';
                            alert('Error: ' + msg);
                            location.reload();
                        }
                    },
                    error: function () {
                        alert('An unexpected server error occurred.');
                        $spinner.removeClass('is-active');
                        $button.prop('disabled', false);
                    }
                });
            });

            // Handle Deactivation
            $(document).on('click', '#aeo-deactivate-license', function (e) {
                e.preventDefault();
                if (!confirm('Are you sure you want to deactivate this license?')) {
                    return;
                }
                var $button = $(this);

                $spinner.addClass('is-active');
                $button.prop('disabled', true);

                $.ajax({
                    url: aeo_ajax_object.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aeo_deactivate_license',
                        nonce: aeo_ajax_object.license_nonce
                    },
                    success: function (response) {
                        alert('License deactivated.');
                        // Refresh usage immediately to flip to Free
                        fetchUsage(true);
                        notifyUsageRefresh();
                        setTimeout(function () { location.reload(); }, 300);
                    },
                    error: function () {
                        alert('An unexpected server error occurred.');
                        $spinner.removeClass('is-active');
                        $button.prop('disabled', false);
                    }
                });
            });

            var analysisPolls = {};
            function stopAnalysisPoll(postId) {
                if (analysisPolls[postId]) {
                    clearInterval(analysisPolls[postId].timer);
                    delete analysisPolls[postId];
                }
            }
            function startAnalysisPoll(postId, jobId, onSuccess, onError) {
                if (!jobId) {
                    if (onError) onError('Missing analysis job.');
                    return;
                }
                stopAnalysisPoll(postId);
                var startedAt = Date.now();
                analysisPolls[postId] = { timer: null, inFlight: false };
                var maxMs = 5 * 60 * 1000;
                var poll = function () {
                    if (!analysisPolls[postId] || analysisPolls[postId].inFlight) {
                        return;
                    }
                    if ((Date.now() - startedAt) > maxMs) {
                        stopAnalysisPoll(postId);
                        if (onError) onError('Analysis timed out. Please try again.');
                        return;
                    }
                    analysisPolls[postId].inFlight = true;
                    $.ajax({
                        url: aeo_ajax_object.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'aeo_poll_analysis_job',
                            post_id: postId,
                            job_id: jobId,
                            nonce: aeo_ajax_object.analyze_nonce
                        },
                        success: function (resp) {
                            if (analysisPolls[postId]) {
                                analysisPolls[postId].inFlight = false;
                            }
                            if (!resp || !resp.success) {
                                var msg = resp && resp.data && resp.data.message ? resp.data.message : 'Analysis failed.';
                                stopAnalysisPoll(postId);
                                if (onError) onError(msg);
                                return;
                            }
                            if (resp.data && resp.data.pending) {
                                return;
                            }
                            stopAnalysisPoll(postId);
                            if (onSuccess) onSuccess(resp.data || {});
                        },
                        error: function () {
                            if (analysisPolls[postId]) {
                                analysisPolls[postId].inFlight = false;
                            }
                            stopAnalysisPoll(postId);
                            if (onError) onError('An unexpected server error occurred. Please try again.');
                        }
                    });
                };
                poll();
                analysisPolls[postId].timer = setInterval(poll, 4000);
            }

            // Analyze button on posts list (first-time)
            $(document).on('click', '.aeo-analyze-button', function (e) {
                if (bulkActive) { e.preventDefault(); return; }
                e.preventDefault();

                var $button = $(this);
                var $rowSpinner = $button.next('.aeo-spinner');
                var postId = $button.data('post-id');

                $button.hide();
                $rowSpinner.addClass('is-active');

                $.ajax({
                    url: aeo_ajax_object.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aeo_analyze_post',
                        post_id: postId,
                        nonce: aeo_ajax_object.analyze_nonce
                    },
                    success: function (response) {
                        if (response && response.success) {
                            if (response.data && response.data.pending) {
                                showAeoNotice(response.data.message ? response.data.message : 'Analysis queued. Waiting for results...', false);
                                notifyUsageRefresh();
                                startAnalysisPoll(postId, response.data.job_id, function () {
                                    showAeoNotice('Post analyzed successfully.', false);
                                    notifyUsageRefresh();
                                    setTimeout(function () { location.reload(); }, 1000);
                                }, function (msg) {
                                    $rowSpinner.removeClass('is-active');
                                    $button.show();
                                    showAeoNotice(msg, true);
                                    notifyUsageRefresh();
                                });
                                return;
                            }
                            showAeoNotice(response.data && response.data.message ? response.data.message : 'Post analyzed successfully.', false);
                            notifyUsageRefresh();
                            setTimeout(function () { location.reload(); }, 1000);
                        } else {
                            $rowSpinner.removeClass('is-active');
                            $button.show();
                            var msg = response && response.data && response.data.message ? response.data.message : 'Analysis failed.';
                            showAeoNotice(msg, true);
                            notifyUsageRefresh();
                        }
                    },
                    error: function () {
                        $rowSpinner.removeClass('is-active');
                        $button.show();
                        showAeoNotice('An unexpected server error occurred. Please try again.', true);
                        notifyUsageRefresh();
                    }
                });
            });

            // Re-analyze icon on posts list (already analyzed posts)
            $(document).on('click', '.aeo-reanalyze-button', function (e) {
                if (bulkActive) { e.preventDefault(); return; }
                e.preventDefault();
                var $link = $(this);
                var postId = $link.data('post-id');
                var $wrapper = $link.closest('.aeo-analyzed-wrapper');
                var $spinnerLocal = $wrapper.length ? $wrapper.find('.aeo-spinner') : $link.closest('.aeo-meta-actions').find('.aeo-meta-spinner');

                // Provide quick visual feedback
                $spinnerLocal.addClass('is-active');
                $link.addClass('aeo-disabled');

                $.ajax({
                    url: aeo_ajax_object.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aeo_analyze_post',
                        post_id: postId,
                        nonce: aeo_ajax_object.analyze_nonce
                    },
                    success: function (response) {
                        if (response && response.success) {
                            if (response.data && response.data.pending) {
                                showAeoNotice(response.data.message ? response.data.message : 'Analysis queued. Waiting for results...', false);
                                notifyUsageRefresh();
                                startAnalysisPoll(postId, response.data.job_id, function () {
                                    showAeoNotice('Post re-analyzed successfully.', false);
                                    notifyUsageRefresh();
                                    setTimeout(function () { location.reload(); }, 1000);
                                }, function (msg) {
                                    $spinnerLocal.removeClass('is-active');
                                    $link.removeClass('aeo-disabled');
                                    showAeoNotice(msg, true);
                                    notifyUsageRefresh();
                                });
                                return;
                            }
                            showAeoNotice(response.data && response.data.message ? response.data.message : 'Post re-analyzed successfully.', false);
                            notifyUsageRefresh();
                            setTimeout(function () { location.reload(); }, 1000); // keep existing reload behavior (manual flows unchanged)
                        } else {
                            $spinnerLocal.removeClass('is-active');
                            $link.removeClass('aeo-disabled');
                            var msg = response && response.data && response.data.message ? response.data.message : 'Re-analysis failed.';
                            showAeoNotice(msg, true);
                            notifyUsageRefresh();
                        }
                    },
                    error: function () {
                        $spinnerLocal.removeClass('is-active');
                        $link.removeClass('aeo-disabled');
                        showAeoNotice('An unexpected server error occurred. Please try again.', true);
                        notifyUsageRefresh();
                    }
                });
            });

            // Note: Header Analyze/Re-analyze button injection removed in favor of the modal FAB/flow.

            /* ===== AJAX Bulk Start (No reload) ===== */
            // Intercept bulk action form submit when our action selected
            var $bulkForm = $('#posts-filter');
            if ($bulkForm.length) {
                $bulkForm.on('submit', function (e) {
                    var actionSelected = '';
                    var primary = $('#bulk-action-selector-top').val();
                    var secondary = $('#bulk-action-selector-bottom').val();
                    if (primary === 'aeo_bulk_analyze') actionSelected = primary; else if (secondary === 'aeo_bulk_analyze') actionSelected = secondary;
                    if (actionSelected !== 'aeo_bulk_analyze') return; // Not our action
                    e.preventDefault();
                    if (bulkActive) return; // Already running
                    var ids = [];
                    $('tbody#the-list tr input[type="checkbox"][name="post[]"]:checked').each(function () { ids.push($(this).val()); });
                    if (!ids.length) { alert('No posts selected.'); return; }
                    // Disable submit to prevent duplicate starts
                    $bulkForm.find('input, select, button').prop('disabled', true);
                    // No progress bar injection per new UX.
                    bulkActive = true;
                    // Mark selected rows immediately (queued / skipped)
                    var selectionMap = {};
                    ids.forEach(function (pid) {
                        var $cell = $('tr#post-' + pid + ' td.column-aeo_analysis');
                        if ($cell.length && !$cell.find('.aeo-bulk-status').length) {
                            $cell.html('<span class="aeo-bulk-status" data-post-id="' + pid + '" data-status="queued"><span class="spinner is-active" style="float:none;margin:0 4px 0 0;"></span><span class="aeo-bulk-status-text">Queued</span></span>');
                        }
                    });
                    $.post(ajaxurl, { action: 'aeo_bulk_start', nonce: aeo_ajax_object.bulk_nonce, post_ids: ids }, function (resp) {
                        if (!resp || !resp.success) {
                            alert((resp && resp.data && resp.data.message) || 'Bulk start failed.');
                            $bulkForm.find('input, select, button').prop('disabled', false);
                            return;
                        }
                        var d = resp.data;
                        // Update any skipped statuses
                        Object.keys(d.statuses).forEach(function (pid) {
                            var st = d.statuses[pid];
                            var $wrap = $('.aeo-bulk-status[data-post-id="' + pid + '"]');
                            if (!$wrap.length) return;
                            if (st === 'skipped') {
                                $wrap.attr('data-status', 'skipped').html('<span class="aeo-bulk-status-text">Skipped</span>');
                            }
                        });
                        // No top progress updates per new UX.
                        bulkNonce = aeo_ajax_object.bulk_nonce; // ensure we have it
                        // Disable manual actions while running
                        $('.aeo-analyze-button, .aeo-reanalyze-button, .aeo-analyze-button-metabox').prop('disabled', true).addClass('disabled');
                        if (d.active) {
                            pollBulk();
                        } else {
                            bulkActive = false;
                            $bulkForm.find('input, select, button').prop('disabled', false);
                            $('.aeo-analyze-button, .aeo-reanalyze-button, .aeo-analyze-button-metabox').prop('disabled', false).removeClass('disabled');
                            showAeoNotice('Bulk analysis completed or could not be started.', false);
                        }
                    }).fail(function () {
                        alert('Bulk start request failed.');
                        $bulkForm.find('input, select, button').prop('disabled', false);
                    });
                });
            }

            // Keep original polling function as-is (defined above) with no progress bar usage.
        });
    }
)(jQuery);
