<?php

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

// Ensure improvements helpers are loaded
require_once plugin_dir_path(__FILE__) . 'improvements.php';

if ( ! function_exists( 'aeo_editor_debug_ui_enabled' ) ) {
    /**
     * Determine whether the editor debug UI should be exposed.
     *
     * @return bool
     */
    function aeo_editor_debug_ui_enabled() {
        $enabled = false;
        if ( defined( 'AEO_DEBUG_EDITOR' ) ) {
            $enabled = (bool) AEO_DEBUG_EDITOR;
        }
        return (bool) apply_filters( 'aeo_editor_debug_ui_enabled', $enabled );
    }
}

// Phase 2: Replace all local free-run counters with server truth via usage-status endpoint

if ( ! function_exists( 'aeo_get_analyzed_index' ) ) {
    /**
     * Retrieve the cached list of analyzed posts keyed by timestamp.
     *
     * @return array<int,int> Map of post_id => unix timestamp.
     */
    function aeo_get_analyzed_index() {
        if ( isset( $GLOBALS['aeo_analyzed_index_cache'] ) && is_array( $GLOBALS['aeo_analyzed_index_cache'] ) ) {
            return $GLOBALS['aeo_analyzed_index_cache'];
        }

        $stored = get_option( 'aeo_analyzed_index', array() );
        if ( ! is_array( $stored ) ) {
            $stored = array();
        }

        $clean = array();
        foreach ( $stored as $post_id => $timestamp ) {
            $pid = (int) $post_id;
            if ( $pid < 1 ) {
                continue;
            }
            $clean[ $pid ] = max( 0, (int) $timestamp );
        }

    $GLOBALS['aeo_analyzed_index_cache'] = $clean;
    return $clean;
    }
}

if ( ! function_exists( 'aeo_update_analyzed_index' ) ) {
    /**
     * Update the analyzed index with the provided post ID.
     *
     * @param int      $post_id   Post being marked analyzed.
     * @param int|null $timestamp Optional unix timestamp. Defaults to now.
     */
    function aeo_update_analyzed_index( $post_id, $timestamp = null ) {
        $post_id   = (int) $post_id;
        $timestamp = ( null === $timestamp ) ? current_time( 'timestamp', true ) : (int) $timestamp;
        if ( $timestamp <= 0 ) {
            $timestamp = current_time( 'timestamp', true );
        }
        if ( $post_id < 1 ) {
            return;
        }

    $index = aeo_get_analyzed_index();
        $index[ $post_id ] = $timestamp;
        arsort( $index );

        update_option( 'aeo_analyzed_index', $index, false );
        update_option( 'aeo_analyzed_index_built', AEO_PLUGIN_VERSION, false );

        // Refresh static cache for future calls within the same request.
    $GLOBALS['aeo_analyzed_index_cache'] = $index;
    }
}

if ( ! function_exists( 'aeo_remove_from_analyzed_index' ) ) {
    /**
     * Remove a post from the analyzed index.
     *
     * @param int $post_id Post ID to remove.
     */
    function aeo_remove_from_analyzed_index( $post_id ) {
        $post_id = (int) $post_id;
        if ( $post_id < 1 ) {
            return;
        }

        $index = aeo_get_analyzed_index();
        if ( isset( $index[ $post_id ] ) ) {
            unset( $index[ $post_id ] );
            update_option( 'aeo_analyzed_index', $index, false );
            $GLOBALS['aeo_analyzed_index_cache'] = $index;
        }
    }
}

if ( ! function_exists( 'aeo_build_analyzed_index' ) ) {
    /**
     * Build the analyzed index by scanning posts for the deterministic meta flag.
     *
     * @param bool $force Whether to rebuild even if a recent index exists.
     * @return array<int,int> Built index map of post_id => timestamp.
     */
    function aeo_build_analyzed_index( $force = false ) {
        $built_version = get_option( 'aeo_analyzed_index_built', '' );
        if ( ! $force && $built_version && version_compare( $built_version, AEO_PLUGIN_VERSION, '>=' ) ) {
            return aeo_get_analyzed_index();
        }

        $index      = array();
        $paged      = 1;
        $chunk_size = 200;

        do {
            $query = new WP_Query( array(
                'post_type'              => 'post',
                'post_status'            => 'any',
                'posts_per_page'         => $chunk_size,
                'paged'                  => $paged,
                'fields'                 => 'ids',
                'orderby'                => 'date',
                'order'                  => 'DESC',
                'no_found_rows'          => true,
                'cache_results'          => false,
                'update_post_term_cache' => false,
            ) );

            if ( empty( $query->posts ) ) {
                break;
            }

            foreach ( $query->posts as $post_id ) {
                $flag = get_post_meta( $post_id, '_aeo_plugin_analyzed', true );
                if ( empty( $flag ) ) {
                    continue;
                }
                $raw_ts = get_post_meta( $post_id, '_aeo_analyzed_at', true );
                $ts     = $raw_ts ? strtotime( $raw_ts ) : false;
                if ( ! $ts ) {
                    $post_date = get_post_field( 'post_modified_gmt', $post_id );
                    $ts        = $post_date ? strtotime( $post_date ) : time();
                }
                $index[ (int) $post_id ] = (int) $ts;
            }

            $paged++;
        } while ( count( $query->posts ) === $chunk_size );

        wp_reset_postdata();

        if ( ! empty( $index ) ) {
            arsort( $index );
        }

        update_option( 'aeo_analyzed_index', $index, false );
        update_option( 'aeo_analyzed_index_built', AEO_PLUGIN_VERSION, false );

        $GLOBALS['aeo_analyzed_index_cache'] = $index;
        return $index;
    }
}

if ( ! function_exists( 'aeo_maybe_build_analyzed_index' ) ) {
    /**
     * Ensure the analyzed index exists for admin views.
     */
    function aeo_maybe_build_analyzed_index() {
        if ( ! is_admin() ) {
            return;
        }
        aeo_build_analyzed_index( false );
    }
}
add_action( 'admin_init', 'aeo_maybe_build_analyzed_index' );

if ( ! function_exists( 'aeo_purge_analyzed_index_on_delete' ) ) {
    /**
     * Remove deleted or trashed posts from the analyzed index.
     *
     * @param int $post_id Post being removed.
     */
    function aeo_purge_analyzed_index_on_delete( $post_id ) {
        aeo_remove_from_analyzed_index( $post_id );
    }
}
add_action( 'before_delete_post', 'aeo_purge_analyzed_index_on_delete' );
add_action( 'trashed_post', 'aeo_purge_analyzed_index_on_delete' );

if ( ! function_exists( 'aeo_mark_post_analyzed' ) ) {
    /**
     * Persist a deterministic flag that indicates the post has finalized AI output.
     *
     * @param int      $post_id   Post ID to mark as analyzed.
     * @param int|null $timestamp Optional unix timestamp to store in the index.
     * @return bool True on success, false if the post ID is invalid.
     */
    function aeo_mark_post_analyzed( $post_id, $timestamp = null ) {
        $post_id = (int) $post_id;
        if ( $post_id < 1 ) {
            return false;
        }

        /**
         * Allow customization of the stored meta value for the analyzed flag.
         *
         * @param string $value   Default value written to the meta key.
         * @param int    $post_id The post being updated.
         */
        $value = apply_filters( 'aeo_analyzed_meta_value', '1', $post_id );

        update_post_meta( $post_id, '_aeo_plugin_analyzed', $value );
        aeo_update_analyzed_index( $post_id, $timestamp );
        return true;
    }
}

if ( ! defined( 'AEO_USAGE_STATUS_API_URL' ) ) {
    // Default to the Supabase edge function. Can be overridden via option 'aeo_status_api_url'.
    define( 'AEO_USAGE_STATUS_API_URL', get_option( 'aeo_status_api_url', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/usage-status' ) );
}

if ( ! function_exists( 'aeo_fetch_usage_status' ) ) {
    function aeo_fetch_usage_status() {
        $api_url = AEO_USAGE_STATUS_API_URL;
        $site_url = home_url();
        $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I';
        // Build payload; include license_key only when WP believes license is active
        $payload = array( 'site_url' => $site_url );
        $license_status = get_option( 'aeo_license_status' );
        $license_key    = get_option( 'aeo_license_key' );
        if ( 'active' === $license_status && ! empty( $license_key ) ) {
            $payload['license_key'] = $license_key;
        }
        $args = array(
            'method'  => 'POST',
            'headers' => array(
                'Authorization' => 'Bearer ' . $supabase_anon_key,
                'apikey'        => $supabase_anon_key,
                'Content-Type'  => 'application/json',
            ),
            'body'    => wp_json_encode( $payload ),
            'timeout' => 20,
        );
        $response = wp_remote_post( $api_url, $args );
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'status_http_error', $response->get_error_message() );
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( 200 !== (int) $code || ! is_array( $data ) ) {
            $msg = isset( $data['message'] ) ? $data['message'] : 'Unable to fetch usage.';
            return new WP_Error( 'status_bad_response', $msg );
        }
        return $data;
    }
}

// Backward-compatible helpers now powered by server truth
if ( ! function_exists( 'aeo_get_free_runs_used' ) ) {
    function aeo_get_free_runs_used() {
        $status = aeo_fetch_usage_status();
        if ( is_wp_error( $status ) ) { return 0; }
        return isset( $status['free']['used'] ) ? max( 0, (int) $status['free']['used'] ) : 0;
    }
}
if ( ! function_exists( 'aeo_get_free_runs_remaining' ) ) {
    function aeo_get_free_runs_remaining() {
        $status = aeo_fetch_usage_status();
        if ( is_wp_error( $status ) ) { return 0; }
        return isset( $status['free']['remaining'] ) ? max( 0, (int) $status['free']['remaining'] ) : 0;
    }
}
if ( ! function_exists( 'aeo_has_free_trial_access' ) ) {
    function aeo_has_free_trial_access() {
        if ( aeo_impr_is_pro_active() ) { return true; }
        return aeo_get_free_runs_remaining() > 0;
    }
}
if ( ! function_exists( 'aeo_get_trial_payload' ) ) {
    function aeo_get_trial_payload() {
        $status = aeo_fetch_usage_status();
        if ( is_wp_error( $status ) ) {
            return array(
                'limit' => 0,
                'used' => 0,
                'remaining' => 0,
                'notice' => __( 'Unable to fetch usage', 'anslift' ),
            );
        }
        $is_pro = ( isset( $status['plan'] ) && $status['plan'] === 'pro' );
        if ( $is_pro ) {
            // Pro users don’t use free preview gating; keep payload neutral
            return array( 'limit' => 0, 'used' => 0, 'remaining' => 0, 'notice' => '' );
        }
        $limit = isset( $status['free']['limit'] ) ? (int) $status['free']['limit'] : 10;
        $used = isset( $status['free']['used'] ) ? (int) $status['free']['used'] : 0;
        $remaining = isset( $status['free']['remaining'] ) ? (int) $status['free']['remaining'] : max( 0, $limit - $used );
        $notice = $remaining > 0
            ? sprintf(
                /* translators: 1: remaining free preview runs, 2: total free preview runs limit */
                __( 'Free preview: %1$d of %2$d runs remaining.', 'anslift' ),
                $remaining,
                $limit
            )
            : __( 'Free preview exhausted. Activate your license to continue.', 'anslift' );
        return array(
            'limit' => $limit,
            'used' => $used,
            'remaining' => $remaining,
            'notice' => $notice,
        );
    }
}
// Deprecated: no-ops maintained for compatibility
if ( ! function_exists( 'aeo_set_free_runs_used' ) ) {
    function aeo_set_free_runs_used( $count ) { return 0; }
}
if ( ! function_exists( 'aeo_increment_free_runs_used' ) ) {
    function aeo_increment_free_runs_used() { return 0; }
}


/**
 * Cleanup all bulk analysis options.
 */
function aeo_bulk_cleanup_options() {
    delete_option('aeo_bulk_active');
    delete_option('aeo_bulk_queue');
    delete_option('aeo_bulk_total');
    delete_option('aeo_bulk_processed');
    delete_option('aeo_bulk_failures');
    delete_option('aeo_bulk_skipped');
    delete_option('aeo_bulk_cancel_requested');
    delete_option('aeo_bulk_started_at');
}

/**
 * Enqueue admin scripts and styles.
 */
function aeo_plugin_enqueue_admin_assets($hook) {
    // Load on: Posts list (edit.php), single post editor (post.php / post-new.php), and the plugin's own admin page (license panel)
    $allowed_hooks = array('edit.php', 'post.php', 'post-new.php', 'toplevel_page_anslift');
    if ( ! in_array( $hook, $allowed_hooks, true ) ) {
        return;
    }

    // Provide a version to avoid browser cache issues and satisfy WP coding standards
    $ver = defined('AEO_PLUGIN_VERSION') ? AEO_PLUGIN_VERSION : false;
    wp_enqueue_style( 'aeo-admin-style', AEO_PLUGIN_URL . 'admin/assets/admin-style.css', array(), $ver );
    wp_enqueue_script( 'aeo-admin-script', AEO_PLUGIN_URL . 'admin/assets/admin-script.js', array( 'jquery' ), $ver, true );

    $current_post_id = 0; $current_post_analyzed = false;
    if ( in_array( $hook, array( 'post.php', 'post-new.php' ), true ) ) {
        $pid = filter_input( INPUT_GET, 'post', FILTER_VALIDATE_INT );
        if ( $pid ) {
            if ( $pid > 0 ) {
                $current_post_id = $pid;
                $current_post_analyzed = (bool) get_post_meta( $pid, '_aeo_plugin_analyzed', true );
            }
        } elseif ( isset( $GLOBALS['post']->ID ) ) {
            $current_post_id = (int) $GLOBALS['post']->ID;
            $current_post_analyzed = (bool) get_post_meta( $current_post_id, '_aeo_plugin_analyzed', true );
        }
    }
    $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
    $ptype  = $screen && isset( $screen->post_type ) ? $screen->post_type : 'post';
    $is_gutenberg = 0;
    if ( function_exists( 'use_block_editor_for_post' ) && $current_post_id ) {
        $is_gutenberg = use_block_editor_for_post( $current_post_id ) ? 1 : 0;
    } elseif ( function_exists( 'use_block_editor_for_post_type' ) ) {
        $is_gutenberg = use_block_editor_for_post_type( $ptype ) ? 1 : 0;
    }
    wp_localize_script( 'aeo-admin-script', 'aeo_ajax_object', array(
        'ajax_url'       => admin_url( 'admin-ajax.php' ),
        'analyze_nonce'  => wp_create_nonce( 'aeo_analyze_nonce' ),
        'license_nonce'  => wp_create_nonce( 'aeo_license_nonce' ),
        'bulk_nonce'     => wp_create_nonce( 'aeo_bulk_nonce' ),
        'status_nonce'   => wp_create_nonce( 'aeo_status_nonce' ),
        'settings_nonce' => wp_create_nonce( 'aeo_settings_nonce' ),
        'timezone'       => get_option('timezone_string') ?: '',
        'current_post_id'=> $current_post_id,
        'current_post_analyzed' => $current_post_analyzed,
        'is_gutenberg'   => $is_gutenberg,
        'labels' => array(
            'analyze'   => __( 'Analyze for AEO', 'anslift' ),
            'reanalyze' => __( 'Re-analyze', 'anslift' ),
            'analyzing' => __( 'Analyzing…', 'anslift' ),
        ),
    ) );

    // Enqueue editor UI scaffold only on single post editor screens
    if ( in_array( $hook, array( 'post.php', 'post-new.php' ), true ) ) {
        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        $ptype  = $screen && isset( $screen->post_type ) ? $screen->post_type : 'post';
        if ( 'post' === $ptype ) {
            wp_enqueue_style( 'aeo-editor-ui-style', AEO_PLUGIN_URL . 'admin/assets/editor-ui.css', array(), '0.1.2' );
            wp_enqueue_script( 'aeo-editor-ui-script', AEO_PLUGIN_URL . 'admin/assets/editor-ui.js', array(), '0.1.2', true );
            $trial_payload = aeo_get_trial_payload();
            $classic_flag_option = get_option( 'aeo_classic_overlays_enabled', 'yes' );
            $classic_overlays_enabled = apply_filters( 'aeo_classic_overlays_enabled', ( 'no' !== strtolower( (string) $classic_flag_option ) ) );
            $classic_overlays_enabled = (bool) $classic_overlays_enabled;
            $debug_ui_enabled = aeo_editor_debug_ui_enabled();

            wp_localize_script( 'aeo-editor-ui-script', 'aeo_editor_object', array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonces' => array(
                    'editor'  => wp_create_nonce( 'aeo_editor_nonce' ),
                    'analyze' => wp_create_nonce( 'aeo_analyze_nonce' ),
                    'license' => wp_create_nonce( 'aeo_license_nonce' ),
                    'bulk'    => wp_create_nonce( 'aeo_bulk_nonce' ),
                    'status'  => wp_create_nonce( 'aeo_status_nonce' ),
                ),
                'post_id' => $current_post_id,
                'is_gutenberg' => $is_gutenberg,
                'license_active' => ( 'active' === get_option( 'aeo_license_status' ) ) ? 1 : 0,
                'current_post_analyzed' => $current_post_analyzed ? 1 : 0,
                'can_analyze' => aeo_has_free_trial_access() ? 1 : 0,
                'trial' => $trial_payload,
                'labels' => array(
                    'open_modal' => __( 'Open AEO Editor', 'anslift' ),
                ),
                'flags' => array(
                    'bulk_active' => (bool) get_option( 'aeo_bulk_active', false ),
                    'classic_overlays_enabled' => $classic_overlays_enabled,
                    'debug_ui' => (bool) $debug_ui_enabled,
                ),
            ) );
        }
    }
}
add_action( 'admin_enqueue_scripts', 'aeo_plugin_enqueue_admin_assets' );

/**
 * Inject the Analyze for AEO button into the post editor header (classic editor).
 * Appears next to the post title area actions. Only show on post type 'post'.
 */
function aeo_inject_editor_header_button() {
    global $pagenow, $post;
    if ( ! in_array( $pagenow, array( 'post.php', 'post-new.php' ), true ) ) { return; }
    $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
    if ( ! $screen || 'post' !== $screen->post_type ) { return; }
    // Avoid duplication if block editor (Gutenberg) where different placement needed (future enhancement)
    if ( function_exists( 'use_block_editor_for_post_type' ) && use_block_editor_for_post_type( 'post' ) ) {
        // Gutenberg active – JS will inject button, skip PHP injection to avoid layout issues.
        return;
    }
    $post_id = isset( $post->ID ) ? (int) $post->ID : 0;
    $is_analyzed = ( $post_id && get_post_meta( $post_id, '_aeo_plugin_analyzed', true ) );
    // Output minimal inline style for alignment (non-invasive)
    echo '<style>.aeo-editor-header-btn{margin-left:8px;}</style>';
    echo '<div class="aeo-editor-header-btn" style="display:inline-block;">';
    if ( ! $is_analyzed ) {
    echo '<button type="button" class="button button-primary aeo-analyze-button-header" data-post-id="' . esc_attr( $post_id ) . '"><span class="dashicons dashicons-search"></span> ' . esc_html__( 'Analyze for AEO', 'anslift' ) . '</button> <span class="spinner aeo-header-spinner" style="float:none;margin-top:0;vertical-align:middle;"></span>';
    } else {
    echo '<button type="button" class="button aeo-reanalyze-button aeo-reanalyze-button-header" data-post-id="' . esc_attr( $post_id ) . '"><span class="dashicons dashicons-update"></span> ' . esc_html__( 'Re-analyze', 'anslift' ) . '</button> <span class="spinner aeo-header-spinner" style="float:none;margin-top:0;vertical-align:middle;"></span>';
    }
    echo '</div>';
}
// Deprecated: modal now contains Analyze; disable header injection to avoid duplicate CTAs.
// add_action( 'in_admin_header', 'aeo_inject_editor_header_button' );


/**
 * Add a custom column to the posts list table.
 */
function aeo_plugin_add_analysis_column( $columns ) {
    $columns['aeo_analysis'] = __( 'AEO Analysis', 'anslift' );
    return $columns;
}
add_filter( 'manage_posts_columns', 'aeo_plugin_add_analysis_column' );

/**
 * Display content for the custom column.
 */
function aeo_plugin_display_analysis_column( $column, $post_id ) {
    if ( 'aeo_analysis' !== $column ) { return; }
    $post_id = (int) $post_id; if ( $post_id < 1 ) { return; }
    $bulk_status = get_post_meta( $post_id, '_aeo_bulk_status', true );
    $bulk_active = get_option( 'aeo_bulk_active', false );

    // If bulk is active show any bulk status; if bulk finished only show problem statuses.
    if ( $bulk_status ) {
        if ( ! $bulk_active && in_array( $bulk_status, array( 'done', 'skipped', 'queued', 'running' ), true ) ) {
            // Treat as normal analyzed state after completion.
            $bulk_status = ''; // fall through to buttons
        }
    }
    if ( $bulk_active && $bulk_status && in_array( $bulk_status, array( 'queued','running','done','skipped','failed','canceled' ), true ) ) {
        $map = array(
            'queued'   => __( 'Queued', 'anslift' ),
            'running'  => __( 'Running…', 'anslift' ),
            'done'     => __( 'Done', 'anslift' ),
            'failed'   => __( 'Failed', 'anslift' ),
            'skipped'  => __( 'Skipped', 'anslift' ),
            'canceled' => __( 'Canceled', 'anslift' ),
        );
        $needs_spinner = in_array( $bulk_status, array( 'queued', 'running' ), true );
        echo '<span class="aeo-bulk-status" data-post-id="' . esc_attr( $post_id ) . '" data-status="' . esc_attr( $bulk_status ) . '">' .
            ( $needs_spinner ? '<span class="spinner is-active" style="float:none;margin:0 4px 0 0;"></span>' : '' ) .
            '<span class="aeo-bulk-status-text">' . esc_html( isset( $map[ $bulk_status ] ) ? $map[ $bulk_status ] : $bulk_status ) . '</span>' .
            '</span>';
        return;
    }
    if ( ! $bulk_active && $bulk_status && in_array( $bulk_status, array( 'failed','canceled' ), true ) ) {
        $map = array(
            'failed'   => __( 'Failed', 'anslift' ),
            'canceled' => __( 'Canceled', 'anslift' ),
        );
        echo '<span class="aeo-bulk-status" data-post-id="' . esc_attr( $post_id ) . '" data-status="' . esc_attr( $bulk_status ) . '">' .
            '<span class="aeo-bulk-status-text">' . esc_html( $map[ $bulk_status ] ) . '</span>' .
            '</span>';
        return;
    }

    // Auto-analysis status handling (only when not in bulk context)
    $auto_status = get_post_meta( $post_id, '_aeo_auto_status', true );
    $analyzed    = (bool) get_post_meta( $post_id, '_aeo_plugin_analyzed', true );

    // While auto-analysis queued/running hide manual buttons & show status indicator
    if ( in_array( $auto_status, array( 'queued', 'running' ), true ) && ! $analyzed ) {
    $label = ( 'queued' === $auto_status ) ? __( 'Queued…', 'anslift' ) : __( 'Analyzing…', 'anslift' );
        echo '<span class="aeo-auto-wrapper" data-post-id="' . esc_attr( $post_id ) . '" data-aeo-auto="' . esc_attr( $auto_status ) . '">'
            . '<span class="spinner is-active" style="float:none;margin:0 4px 0 0;"></span>'
            . '<span class="aeo-auto-status-text">' . esc_html( $label ) . '</span>'
            . '</span>';
        return;
    }

    // If auto failed (after retry) & still not analyzed: show manual Analyze button again
    if ( 'failed' === $auto_status && ! $analyzed ) {
    echo '<button type="button" class="button aeo-analyze-button" data-post-id="' . esc_attr( $post_id ) . '">' . esc_html__( 'Analyze for AEO', 'anslift' ) . '</button><div class="aeo-spinner spinner"></div>';
        return;
    }

    // Normal buttons (reanalyze vs analyze)
    if ( $analyzed ) {
    echo '<span class="aeo-analyzed-wrapper"><a href="#" class="button aeo-reanalyze-button" data-post-id="' . esc_attr( $post_id ) . '"><span class="dashicons dashicons-update aeo-reanalyze-icon"></span><span>' . esc_html__( 'Re-analyze', 'anslift' ) . '</span></a><div class="aeo-spinner spinner"></div></span>';
    } else {
    echo '<button type="button" class="button aeo-analyze-button" data-post-id="' . esc_attr( $post_id ) . '">' . esc_html__( 'Analyze for AEO', 'anslift' ) . '</button><div class="aeo-spinner spinner"></div>';
    }
}
add_action( 'manage_posts_custom_column', 'aeo_plugin_display_analysis_column', 10, 2 );

// Lightweight polling script (admin footer on post list) to refresh analysis column for queued/running states.
function aeo_admin_post_list_polling_script() {
    // Only on Posts list
    global $pagenow, $wp_query;
    if ( 'edit.php' !== $pagenow ) { return; }
    $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
    if ( $screen && isset( $screen->post_type ) && 'post' !== $screen->post_type ) { return; }
    $needs = false;
    if ( $wp_query && ! empty( $wp_query->posts ) ) {
        foreach ( $wp_query->posts as $p ) {
            $st = get_post_meta( $p->ID, '_aeo_auto_status', true );
            if ( in_array( $st, array( 'queued', 'running' ), true ) ) { $needs = true; break; }
        }
    }
    if ( ! $needs ) { return; }
    ?>
    <script>
    (function($){
        const nonce = '<?php echo esc_js( wp_create_nonce('aeo_analyze_nonce') ); ?>';
        const pollInterval = 6000; // 6s per requirement
        const shownToasts = new Set();
        function showAeoToast(msg){
            var noticeHtml = '<div class="aeo-notice updated"><p>'+msg+'</p></div>';
            var $n = $(noticeHtml).appendTo('body').fadeIn();
            setTimeout(function(){ $n.fadeOut(function(){ $(this).remove(); }); }, 4000);
        }
        function collectIds(){
            const ids = [];
            $('td.column-aeo_analysis .aeo-auto-wrapper').each(function(){ ids.push($(this).data('post-id')); });
            return ids;
        }
        function poll(){
            const ids = collectIds(); if(!ids.length){ return; }
            $.post(ajaxurl,{action:'aeo_poll_auto_analysis', ids: ids, nonce: nonce}, function(resp){
                if(!resp || !resp.success || !resp.data){ return; }
                const map = resp.data.statuses || {};
                Object.keys(map).forEach(function(pid){
                    const st = map[pid];
                    const $wrap = $('td.column-aeo_analysis .aeo-auto-wrapper[data-post-id="'+pid+'"]');
                    if(!$wrap.length) return;
                    if(st==='queued'){
                        $wrap.attr('data-aeo-auto','queued').find('.aeo-auto-status-text').text('<?php echo esc_js(__('Queued…','anslift')); ?>');
                    } else if(st==='running'){
                        $wrap.attr('data-aeo-auto','running').find('.aeo-auto-status-text').text('<?php echo esc_js(__('Analyzing…','anslift')); ?>');
                    } else if(st==='done'){
                        // Replace with re-analyze button & toast (once)
                        $wrap.replaceWith('<span class="aeo-analyzed-wrapper"><a href="#" class="button aeo-reanalyze-button" data-post-id="'+pid+'"><span class="dashicons dashicons-update aeo-reanalyze-icon"></span><span><?php echo esc_js(__('Re-analyze','anslift')); ?></span></a><div class="aeo-spinner spinner"></div></span>');
                        if(!shownToasts.has(pid)){
                            const rowTitle = $('tr#post-'+pid+' .row-title').first().text() || $('tr#post-'+pid+' .column-title a').first().text() || 'Post';
                            showAeoToast('<?php echo esc_js(__('AEO analysis completed for','anslift')); ?> "'+rowTitle+'".');
                            shownToasts.add(pid);
                        }
                    } else if(st==='failed'){
                        // Replace with manual analyze button
                        $wrap.replaceWith('<button type="button" class="button aeo-analyze-button" data-post-id="'+pid+'"><?php echo esc_js(__('Analyze for AEO','anslift')); ?></button><div class="aeo-spinner spinner"></div>');
                    }
                });
            });
        }
        poll();
        setInterval(poll, pollInterval);
        window.addEventListener('focus', poll);
    })(jQuery);
    </script>
    <?php
}
add_action( 'admin_footer', 'aeo_admin_post_list_polling_script' );

// AJAX polling endpoint
function aeo_poll_auto_analysis() {
    check_ajax_referer( 'aeo_analyze_nonce', 'nonce' );
    if ( ! current_user_can( 'edit_posts' ) ) { wp_send_json_error(); }
    // Sanitize incoming IDs array from POST without directly accessing $_POST
    $ids_input = filter_input( INPUT_POST, 'ids', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
    if ( ! is_array( $ids_input ) ) { $ids_input = array(); }
    $ids = array_map( 'absint', $ids_input );
    $out = array();
    foreach ( $ids as $pid ) {
        if ( $pid < 1 ) { continue; }
        $status = get_post_meta( $pid, '_aeo_auto_status', true );
        // Infer completion if meta indicates analyzed but status not updated
        if ( 'done' !== $status && get_post_meta( $pid, '_aeo_plugin_analyzed', true ) ) {
            $status = 'done';
            update_post_meta( $pid, '_aeo_auto_status', 'done' );
        }
        $out[ $pid ] = $status;
    }
    wp_send_json_success( [ 'statuses' => $out ] );
}
add_action( 'wp_ajax_aeo_poll_auto_analysis', 'aeo_poll_auto_analysis' );


if ( ! function_exists( 'aeo_build_analysis_response_payload' ) ) {
    function aeo_build_analysis_response_payload( $post_id, $result ) {
        $post_id = (int) $post_id;
        $is_pro = aeo_impr_is_pro_active();

        $overview = isset( $result['overview'] ) ? (string) $result['overview'] : (string) get_post_meta( $post_id, '_aeo_ai_overview', true );
        $faqs     = isset( $result['faqs'] ) ? (array) $result['faqs'] : (array) get_post_meta( $post_id, '_aeo_faqs', true );

        if ( isset( $result['improvements'] ) && is_array( $result['improvements'] ) ) {
            $current_impr = $result['improvements'];
        } else {
            $current_impr = aeo_impr_get( $post_id );
        }

        if ( isset( $result['improvements_count'] ) && is_numeric( $result['improvements_count'] ) ) {
            $improvements_count = max( 0, (int) $result['improvements_count'] );
        } else {
            $improvements_count = is_array( $current_impr ) ? count( $current_impr ) : 0;
        }
        $improvements_pending = $improvements_count > 0;
        $trial_payload        = aeo_get_trial_payload();
        $has_trial_remaining  = ! empty( $trial_payload['remaining'] );
        $improvements_locked  = ( ! $is_pro && ! $has_trial_remaining && $improvements_count === 0 );

        $payload = [
            'message'               => __( 'Post analyzed and updated successfully!', 'anslift' ),
            'overview'              => $overview,
            'faqs'                  => $faqs,
            'analyzed'              => true,
            'analyzed_at'           => (string) get_post_meta( $post_id, '_aeo_analyzed_at', true ),
            'improvements_count'    => $improvements_count,
            'improvements_pending'  => $improvements_pending,
            'improvements_pro_only' => $improvements_locked,
            'improvements'          => is_array( $current_impr ) ? $current_impr : array(),
            'license_active'        => (bool) $is_pro,
            'trial'                 => $trial_payload,
        ];
        if ( aeo_editor_debug_ui_enabled() ) {
            $payload['debug_log'] = get_post_meta( $post_id, '_aeo_debug_log', true );
        }
        return $payload;
    }
}

if ( ! function_exists( 'aeo_enqueue_analysis_job' ) ) {
    function aeo_enqueue_analysis_job( $post_id, $segments = null ) {
        if ( ! $post_id || 'post' !== get_post_type( $post_id ) ) {
            return new WP_Error( 'invalid_post', 'Invalid post for analysis.' );
        }
        $post_obj = get_post( $post_id );
        if ( ! $post_obj ) {
            return new WP_Error( 'missing_post', 'Post not found.' );
        }
        $license_key    = get_option( 'aeo_license_key' );
        $license_status = get_option( 'aeo_license_status' );
        $clean = wp_strip_all_tags( strip_shortcodes( $post_obj->post_content ) );
        $clean = preg_replace( '/\s+/', ' ', $clean );
        $max_chars = apply_filters( 'aeo_analysis_max_chars', 18000 );
        if ( strlen( $clean ) > $max_chars ) { $clean = substr( $clean, 0, $max_chars ); }
        if ( ! defined( 'AEO_API_URL' ) ) {
            define( 'AEO_API_URL', get_option( 'aeo_api_url', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/analyze-content' ) );
        }
        $api_url = AEO_API_URL;
        $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I';

        $body = [
            'job_action'   => 'enqueue',
            'site_url'     => home_url(),
            'post_id'      => $post_id,
            'post_title'   => $post_obj->post_title,
            'post_content' => $clean,
            'mode'         => 'analyze+improve',
        ];

        // Attach segments if provided, else attempt server-side segmentation for non-editor flows
        $seg_count = ( ! empty( $segments ) && ! empty( $segments['items'] ) && is_array( $segments['items'] ) ) ? count( $segments['items'] ) : 0;
        $should_fallback = ( $seg_count > 0 && $seg_count <= 1 && strlen( $clean ) > 800 );
        if ( empty( $segments ) || empty( $segments['items'] ) || $should_fallback ) {
            $seg = aeo_impr_segment_post_content( $post_id );
            if ( ! empty( $seg['items'] ) ) {
                $body['segments'] = $seg;
            }
        } else {
            $body['segments'] = $segments;
        }
        if ( ! empty( $license_key ) && 'active' === $license_status ) { $body['license_key'] = $license_key; }

        $body['content_hash'] = hash( 'sha256', $clean );
        if ( ! empty( $body['segments'] ) ) {
            $body['segments_hash'] = hash( 'sha256', wp_json_encode( $body['segments'] ) );
        }

        $timeout = (int) apply_filters( 'aeo_analysis_queue_timeout', 15 );
        $args = [
            'method'  => 'POST',
            'headers' => [
                'Authorization' => 'Bearer ' . $supabase_anon_key,
                'apikey'        => $supabase_anon_key,
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode( $body ),
            'timeout' => $timeout,
        ];
        $response = wp_remote_post( $api_url, $args );
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'queue_failed', $response->get_error_message() );
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body_raw = wp_remote_retrieve_body( $response );
        $data = json_decode( $body_raw, true );
        if ( 200 !== $code ) {
            $msg = is_array( $data ) && isset( $data['message'] ) ? $data['message'] : 'Queue request failed.';
            return new WP_Error( 'queue_failed', $msg );
        }
        if ( ! is_array( $data ) || empty( $data['job_id'] ) ) {
            return new WP_Error( 'queue_failed', 'Queue response missing job ID.' );
        }
        return $data;
    }
}

/**
 * Handle the AJAX request to analyze a post.
 */
function aeo_plugin_handle_analyze_post() {
    check_ajax_referer('aeo_analyze_nonce', 'nonce');

    if ( empty( $_POST['post_id'] ) ) {
        wp_send_json_error(['message' => 'Error: Missing Post ID.']);
        return;
    }

    $post_id = absint( wp_unslash( $_POST['post_id'] ) );

    if (!current_user_can('edit_post', $post_id)) {
        wp_send_json_error(['message' => 'Error: You do not have permission to edit this post.']);
        return;
    }
    $post_status = get_post_status( $post_id );
    if ( ! $post_status ) {
        wp_send_json_error(['message' => 'Error: Post not found.']);
        return;
    }
    if ( 'auto-draft' === $post_status ) {
        wp_send_json_error(['message' => __( 'Save the Post before analyzing', 'anslift' )]);
        return;
    }

    // Optional: parse segments provided by the editor (Gutenberg/Classic)
    // Read as array first; if not array, fall back to raw string
    $segments_array = filter_input( INPUT_POST, 'segments', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
    if ( is_array( $segments_array ) ) {
        $segments_from_request = aeo_impr_parse_segments_from_request( $segments_array );
    } else {
        $segments_raw = filter_input( INPUT_POST, 'segments', FILTER_UNSAFE_RAW );
        $segments_from_request = is_string( $segments_raw ) ? aeo_impr_parse_segments_from_request( $segments_raw ) : null;
    }

    $is_pro = aeo_impr_is_pro_active();
    // Phase 2: Do not pre-gate or increment locally. Trust server quotas (402/429) and reflect results in UI.

    $enqueue = aeo_enqueue_analysis_job( $post_id, $segments_from_request );
    if ( is_wp_error( $enqueue ) ) {
        $error_payload = array( 'message' => $enqueue->get_error_message() );
        $error_payload['trial'] = aeo_get_trial_payload();
        wp_send_json_error( $error_payload );
    }

    $job_id = sanitize_text_field( $enqueue['job_id'] );
    $status = isset( $enqueue['status'] ) ? sanitize_key( $enqueue['status'] ) : 'queued';
    if ( ! in_array( $status, array( 'queued', 'running', 'done' ), true ) ) {
        $status = 'queued';
    }
    update_post_meta( $post_id, '_aeo_analysis_job_id', $job_id );
    update_post_meta( $post_id, '_aeo_auto_status', ( 'running' === $status ? 'running' : 'queued' ) );

    if ( empty( $enqueue['started'] ) ) {
        wp_schedule_single_event( time() + 5, 'aeo_process_analysis_job_event', array( $post_id, $job_id ) );
    }

    wp_send_json_success( [
        'pending'        => true,
        'status'         => $status,
        'job_id'         => $job_id,
        'message'        => __( 'Analysis queued. Waiting for results...', 'anslift' ),
        'license_active' => (bool) $is_pro,
        'trial'          => aeo_get_trial_payload(),
    ] );
}
add_action( 'wp_ajax_aeo_analyze_post', 'aeo_plugin_handle_analyze_post' );

if ( ! function_exists( 'aeo_fetch_analysis_job_status' ) ) {
    function aeo_fetch_analysis_job_status( $post_id, $job_id ) {
        $post_id = (int) $post_id;
        if ( ! $post_id || empty( $job_id ) ) {
            return new WP_Error( 'missing_job', 'Missing analysis job details.' );
        }
        if ( ! defined( 'AEO_API_URL' ) ) {
            define( 'AEO_API_URL', get_option( 'aeo_api_url', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/analyze-content' ) );
        }
        $api_url = AEO_API_URL;
        $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I';
        $body = [
            'job_action' => 'consume',
            'site_url'   => home_url(),
            'post_id'    => $post_id,
            'job_id'     => $job_id,
        ];
        $timeout = (int) apply_filters( 'aeo_analysis_poll_timeout', 20 );
        $args = [
            'method'  => 'POST',
            'headers' => [
                'Authorization' => 'Bearer ' . $supabase_anon_key,
                'apikey'        => $supabase_anon_key,
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode( $body ),
            'timeout' => $timeout,
        ];
        $response = wp_remote_post( $api_url, $args );
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'job_fetch_failed', $response->get_error_message() );
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body_raw = wp_remote_retrieve_body( $response );
        $data = json_decode( $body_raw, true );
        if ( 404 === (int) $code ) {
            return array( 'status' => 'missing' );
        }
        if ( 200 !== $code ) {
            $msg = is_array( $data ) && isset( $data['message'] ) ? $data['message'] : 'Fetch request failed.';
            return new WP_Error( 'job_fetch_failed', $msg );
        }
        if ( ! is_array( $data ) || empty( $data['status'] ) ) {
            return new WP_Error( 'job_fetch_failed', 'Invalid analysis job response.' );
        }
        return $data;
    }
}

function aeo_process_analysis_job_event_cb( $post_id, $job_id ) {
    $post_id = absint( $post_id );
    $job_id = sanitize_text_field( $job_id );
    if ( $post_id < 1 || empty( $job_id ) ) { return; }
    if ( ! get_post( $post_id ) ) { return; }
    if ( ! defined( 'AEO_API_URL' ) ) {
        define( 'AEO_API_URL', get_option( 'aeo_api_url', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/analyze-content' ) );
    }
    $api_url = AEO_API_URL;
    $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I';
    $body = [
        'job_action' => 'process',
        'site_url'   => home_url(),
        'post_id'    => $post_id,
        'job_id'     => $job_id,
    ];
    $timeout = (int) apply_filters( 'aeo_analysis_process_timeout', 90 );
    wp_remote_post( $api_url, [
        'method'  => 'POST',
        'headers' => [
            'Authorization' => 'Bearer ' . $supabase_anon_key,
            'apikey'        => $supabase_anon_key,
            'Content-Type'  => 'application/json',
        ],
        'body'    => wp_json_encode( $body ),
        'timeout' => $timeout,
    ] );
}
add_action( 'aeo_process_analysis_job_event', 'aeo_process_analysis_job_event_cb', 10, 2 );

function aeo_poll_analysis_job() {
    check_ajax_referer( 'aeo_analyze_nonce', 'nonce' );
    $post_id = absint( filter_input( INPUT_POST, 'post_id', FILTER_SANITIZE_NUMBER_INT ) );
    $job_id  = sanitize_text_field( (string) filter_input( INPUT_POST, 'job_id', FILTER_UNSAFE_RAW ) );

    if ( $post_id < 1 || empty( $job_id ) ) {
        wp_send_json_error( [ 'message' => 'Missing analysis job.' ] );
    }
    if ( ! current_user_can( 'edit_post', $post_id ) ) {
        wp_send_json_error( [ 'message' => 'Error: You do not have permission to edit this post.' ] );
    }

    $data = aeo_fetch_analysis_job_status( $post_id, $job_id );
    if ( is_wp_error( $data ) ) {
        wp_send_json_error( [ 'message' => $data->get_error_message() ] );
    }

    $status = sanitize_key( $data['status'] );
    if ( in_array( $status, array( 'queued', 'running' ), true ) ) {
        update_post_meta( $post_id, '_aeo_auto_status', $status );
        wp_send_json_success( [
            'pending' => true,
            'status'  => $status,
            'job_id'  => $job_id,
        ] );
    }

    if ( 'error' === $status ) {
        update_post_meta( $post_id, '_aeo_auto_status', 'failed' );
        delete_post_meta( $post_id, '_aeo_analysis_job_id' );
        $msg = isset( $data['message'] ) ? (string) $data['message'] : 'Analysis failed.';
        wp_send_json_error( [ 'message' => $msg ] );
    }

    if ( 'missing' === $status ) {
        if ( get_post_meta( $post_id, '_aeo_plugin_analyzed', true ) ) {
            update_post_meta( $post_id, '_aeo_auto_status', 'done' );
            delete_post_meta( $post_id, '_aeo_analysis_job_id' );
            wp_send_json_success( aeo_build_analysis_response_payload( $post_id, array() ) );
        }
        update_post_meta( $post_id, '_aeo_auto_status', 'failed' );
        delete_post_meta( $post_id, '_aeo_analysis_job_id' );
        wp_send_json_error( [ 'message' => 'Analysis job expired or missing.' ] );
    }

    if ( 'done' !== $status ) {
        wp_send_json_error( [ 'message' => 'Analysis status unknown.' ] );
    }

    $payload = isset( $data['payload'] ) && is_array( $data['payload'] ) ? $data['payload'] : null;
    if ( empty( $payload ) ) {
        update_post_meta( $post_id, '_aeo_auto_status', 'failed' );
        delete_post_meta( $post_id, '_aeo_analysis_job_id' );
        wp_send_json_error( [ 'message' => 'Analysis completed without payload.' ] );
    }

    $debug_log = array(
        'timestamp' => current_time( 'mysql' ),
        'post_id'   => $post_id,
        'job_id'    => $job_id,
        'request'   => array( 'source' => 'async_poll' ),
        'response'  => array(
            'data_keys' => array_keys( $payload ),
        ),
        'mapping'   => array(),
    );
    $result = aeo_apply_analysis_payload( $post_id, $payload, null, $debug_log );
    if ( is_wp_error( $result ) ) {
        update_post_meta( $post_id, '_aeo_auto_status', 'failed' );
        delete_post_meta( $post_id, '_aeo_analysis_job_id' );
        wp_send_json_error( [ 'message' => $result->get_error_message() ] );
    }

    // Success: clear manual edits marker
    delete_post_meta( $post_id, '_aeo_manual_edits' );
    delete_post_meta( $post_id, '_aeo_analysis_job_id' );
    update_post_meta( $post_id, '_aeo_auto_status', 'done' );

    wp_send_json_success( aeo_build_analysis_response_payload( $post_id, $result ) );
}
add_action( 'wp_ajax_aeo_poll_analysis_job', 'aeo_poll_analysis_job' );

/**
 * AJAX: Proxy usage-status from backend for admin-only requests.
 */
function aeo_ajax_usage_status() {
    check_ajax_referer( 'aeo_status_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( array( 'message' => 'Forbidden' ), 403 ); }
    $status = aeo_fetch_usage_status();
    if ( is_wp_error( $status ) ) {
        wp_send_json_error( array( 'message' => $status->get_error_message() ), 500 );
    }
    wp_send_json_success( $status );
}
add_action( 'wp_ajax_aeo_usage_status', 'aeo_ajax_usage_status' );

/**
 * Reusable analyzer: performs API request & stores meta.
 */
if ( ! function_exists( 'aeo_apply_analysis_payload' ) ) {
    function aeo_apply_analysis_payload( $post_id, $data, $segments = null, &$debug_log = null ) {
        $post_id = (int) $post_id;
        if ( ! $post_id || 'post' !== get_post_type( $post_id ) ) {
            return new WP_Error( 'invalid_post', 'Invalid post for analysis.' );
        }
        $post_obj = get_post( $post_id );
        if ( ! $post_obj ) {
            return new WP_Error( 'missing_post', 'Post not found.' );
        }

        $rcv_count = 0;
        $keys = array();
        $overview = '';
        $faqs = array();
        $improvements = array();

        if ( isset( $data['ai_overview'], $data['faqs'] ) ) {
            $overview = sanitize_textarea_field( $data['ai_overview'] );
            $faqs = $data['faqs'];
            $improvements = isset( $data['improvements'] ) && is_array( $data['improvements'] ) ? $data['improvements'] : array();
            $rcv_count = is_array( $improvements ) ? count( $improvements ) : 0;
            $keys = array_keys( $data );
        } elseif ( isset( $data['choices'][0]['message']['content'] ) ) {
            $gc_json = $data['choices'][0]['message']['content'];
            $gc = json_decode( $gc_json, true );
            if ( ! isset( $gc['ai_overview'], $gc['faqs'] ) ) {
                if ( is_array( $debug_log ) ) {
                    $debug_log['response']['error'] = 'Invalid JSON structure in choices';
                    update_post_meta( $post_id, '_aeo_debug_log', $debug_log );
                }
                return new WP_Error( 'invalid_json', 'Invalid JSON format.' );
            }
            $overview = sanitize_textarea_field( $gc['ai_overview'] );
            $faqs = $gc['faqs'];
            $improvements = isset( $gc['improvements'] ) && is_array( $gc['improvements'] ) ? $gc['improvements'] : array();
            $rcv_count = is_array( $improvements ) ? count( $improvements ) : 0;
            $keys = array_keys( $gc );
        } else {
            if ( is_array( $debug_log ) ) {
                $debug_log['response']['error'] = 'Unexpected response structure';
                update_post_meta( $post_id, '_aeo_debug_log', $debug_log );
            }
            return new WP_Error( 'unexpected_response', 'Unexpected API response.' );
        }

        if ( is_array( $debug_log ) ) {
            $debug_log['response']['parsed'] = array(
                'overview_len' => strlen( $overview ),
                'faqs_count' => count( $faqs ),
                'improvements_count' => $rcv_count,
                'improvements_raw' => $improvements,
            );
        }

        $analyzed_at = current_time( 'mysql' );
        update_post_meta( $post_id, '_aeo_ai_overview', $overview );
        update_post_meta( $post_id, '_aeo_faqs', $faqs );
        update_post_meta( $post_id, '_aeo_analyzed_at', $analyzed_at );
        $timestamp_gmt = strtotime( get_gmt_from_date( $analyzed_at ) );
        aeo_mark_post_analyzed( $post_id, $timestamp_gmt );
        $stored_improvements = array();

        // Store improvements when provided
        if ( ! empty( $improvements ) && is_array( $improvements ) ) {
            $items = array();
            // Build a segmentation snapshot of current post content to compute baseline hashes
            $seg_snapshot = ( ! empty( $segments ) && ! empty( $segments['items'] ) ) ? $segments : aeo_impr_segment_post_content( $post_id );
            $seg_map = array(); // index => [ 'text' => ..., 'blockType' => ... ]
            if ( isset( $seg_snapshot['items'] ) && is_array( $seg_snapshot['items'] ) ) {
                foreach ( $seg_snapshot['items'] as $seg_item ) {
                    if ( isset( $seg_item['index'] ) ) {
                        $seg_map[ intval( $seg_item['index'] ) ] = array(
                            'text'      => isset( $seg_item['text'] ) ? (string) $seg_item['text'] : '',
                            'blockType' => isset( $seg_item['blockType'] ) ? (string) $seg_item['blockType'] : '',
                            'hints'     => isset( $seg_item['hints'] ) && is_array( $seg_item['hints'] ) ? $seg_item['hints'] : array(),
                        );
                    }
                }
            }
            $similarity_score = function( $a, $b ) {
                $a = aeo_impr_normalize_text( $a );
                $b = aeo_impr_normalize_text( $b );
                if ( $a === '' || $b === '' ) { return 0; }
                $pct = 0;
                similar_text( $a, $b, $pct );
                return $pct;
            };
            foreach ( $improvements as $imp ) {
                if ( ! is_array( $imp ) ) { continue; }
                $idx  = isset( $imp['index'] ) ? intval( $imp['index'] ) : null;
                $orig = isset( $imp['original_text'] ) ? (string) $imp['original_text'] : '';
                $sugg = isset( $imp['suggested_text'] ) ? (string) $imp['suggested_text'] : '';
                $sugg_html = isset( $imp['suggested_html'] ) ? (string) $imp['suggested_html'] : '';
                if ( '' === $sugg && '' !== $sugg_html ) {
                    $sugg = wp_strip_all_tags( $sugg_html );
                }
                if ( '' === $orig || ( '' === $sugg && '' === $sugg_html ) ) { continue; }
                $orig_norm = aeo_impr_normalize_text( $orig );
                if ( null === $idx && $orig_norm !== '' && ! empty( $seg_map ) ) {
                    foreach ( $seg_map as $seg_idx => $seg_item ) {
                        if ( $orig_norm === aeo_impr_normalize_text( $seg_item['text'] ) ) {
                            $idx = $seg_idx;
                            break;
                        }
                    }
                    if ( null === $idx ) {
                        $best_idx = null;
                        $best_score = 0;
                        foreach ( $seg_map as $seg_idx => $seg_item ) {
                            $score = $similarity_score( $orig_norm, $seg_item['text'] );
                            if ( $score > $best_score ) {
                                $best_score = $score;
                                $best_idx = $seg_idx;
                            }
                        }
                        if ( null !== $best_idx && $best_score >= 98 ) {
                            $idx = $best_idx;
                        }
                    }
                }
                if ( null === $idx || ! isset( $seg_map[ $idx ] ) ) { continue; }
                // Prefer hashing the actual segmented content at this index to avoid model drift
                $baseline_text = (string) $seg_map[ $idx ]['text'];
                $block_type    = (string) $seg_map[ $idx ]['blockType'];
                $hints = array();
                if ( isset( $seg_map[ $idx ]['hints'] ) && is_array( $seg_map[ $idx ]['hints'] ) ) {
                    $hints = $seg_map[ $idx ]['hints'];
                }
                if ( empty( $hints ) ) {
                    $hints = aeo_impr_detect_structural_hints( $baseline_text );
                }
                $baseline_match = $similarity_score( $orig_norm, $baseline_text );
                if ( $orig_norm !== '' && $baseline_match < 98 ) { continue; }
                $structure_type = isset( $imp['structure_type'] ) ? strtolower( (string) $imp['structure_type'] ) : 'paragraph';
                if ( $structure_type === 'ordered_list' ) { $structure_type = 'numbered_list'; }
                if ( $structure_type === 'unordered_list' || $structure_type === 'bullet_list' || $structure_type === 'bullets' ) { $structure_type = 'bulleted_list'; }
                $has_ul = ( $sugg_html !== '' && preg_match( '/<\s*ul\b/i', $sugg_html ) );
                $has_ol = ( $sugg_html !== '' && preg_match( '/<\s*ol\b/i', $sugg_html ) );
                $has_table = ( $sugg_html !== '' && preg_match( '/<\s*table\b/i', $sugg_html ) );
                if ( $has_ul || $has_ol || $has_table ) {
                    $structure_type = $has_table ? 'table' : ( $has_ol ? 'numbered_list' : 'bulleted_list' );
                }
                if ( ! in_array( $structure_type, array( 'paragraph', 'bulleted_list', 'numbered_list', 'table' ), true ) ) {
                    $structure_type = 'paragraph';
                }
                if ( $structure_type !== 'paragraph' && $sugg_html === '' ) {
                    $structure_type = 'paragraph';
                }
                $apply_mode = 'auto';
                $locator = array( 'index' => $idx, 'hash' => aeo_impr_hash_text( $baseline_text ) );
                if ( $block_type ) {
                    $locator['blockType'] = $block_type;
                }

                if ( is_array( $debug_log ) ) {
                    $debug_log['mapping'][] = array(
                        'imp_index' => $idx,
                        'orig_text_sample' => substr( $orig, 0, 50 ),
                        'baseline_text_sample' => substr( $baseline_text, 0, 50 ),
                        'hash' => $locator['hash'],
                        'block_type' => $block_type,
                        'match_found' => ( $baseline_text === $orig ) ? 'exact' : 'fuzzy/none',
                    );
                }

                $items[] = aeo_impr_sanitize_item( array(
                    'editor'        => aeo_impr_is_gutenberg_content( $post_obj->post_content ) ? 'gutenberg' : 'classic',
                    'locator'       => $locator,
                    'originalText'  => $baseline_text,
                    'suggestedText' => $sugg,
                    'suggestedHtml' => $sugg_html,
                    'structureType' => $structure_type,
                    'applyMode'     => $apply_mode,
                    'score'         => isset( $imp['score'] ) ? floatval( $imp['score'] ) : 0,
                    'reason'        => isset( $imp['reason'] ) ? (string) $imp['reason'] : '',
                ) );
            }
            // Overwrite existing pending suggestions with new set per spec
            $stored = aeo_impr_overwrite( $post_id, $items );
            $stored_improvements = is_array( $stored ) ? $stored : array();
        } else {
            // Ensure we clear out stale improvements if none were returned
            aeo_impr_clear( $post_id );
        }

        if ( is_array( $debug_log ) ) {
            update_post_meta( $post_id, '_aeo_debug_log', $debug_log );
        }

        // Force status to done to clear any lingering queued/running states
        $curr_status = get_post_meta( $post_id, '_aeo_auto_status', true );
        if ( 'done' !== $curr_status ) {
            update_post_meta( $post_id, '_aeo_auto_status', 'done' );
        }

        if ( empty( $stored_improvements ) ) {
            $stored_improvements = aeo_impr_get( $post_id );
        }

        return [
            'overview'      => $overview,
            'faqs'          => $faqs,
            'improvements'  => is_array( $stored_improvements ) ? $stored_improvements : array(),
            'improvements_count' => is_array( $stored_improvements ) ? count( $stored_improvements ) : 0,
        ];
    }
}

if ( ! function_exists( 'aeo_perform_analysis' ) ) {
    function aeo_perform_analysis( $post_id, $overwrite = true, $segments = null ) {
        if ( ! $post_id || 'post' !== get_post_type( $post_id ) ) {
            return new WP_Error( 'invalid_post', 'Invalid post for analysis.' );
        }
        $post_obj = get_post( $post_id );
        if ( ! $post_obj ) {
            return new WP_Error( 'missing_post', 'Post not found.' );
        }
        if ( ! $overwrite && get_post_meta( $post_id, '_aeo_plugin_analyzed', true ) ) {
            return new WP_Error( 'already_analyzed', 'Post already analyzed.' );
        }
        $license_key    = get_option( 'aeo_license_key' );
        $license_status = get_option( 'aeo_license_status' );
        $clean = wp_strip_all_tags( strip_shortcodes( $post_obj->post_content ) );
        $clean = preg_replace( '/\s+/', ' ', $clean );
        $max_chars = apply_filters( 'aeo_analysis_max_chars', 18000 );
        if ( strlen( $clean ) > $max_chars ) { $clean = substr( $clean, 0, $max_chars ); }
        if ( ! defined( 'AEO_API_URL' ) ) {
            define( 'AEO_API_URL', get_option( 'aeo_api_url', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/analyze-content' ) );
        }
        $api_url = AEO_API_URL;
        $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I';
        $body = [
            'site_url'     => home_url(),
            'post_title'   => $post_obj->post_title,
            'post_content' => $clean,
            'mode'         => 'analyze+improve',
        ];
        // Attach segments if provided, else attempt server-side segmentation for non-editor flows
        $seg_count = ( ! empty( $segments ) && ! empty( $segments['items'] ) && is_array( $segments['items'] ) ) ? count( $segments['items'] ) : 0;
        $should_fallback = ( $seg_count > 0 && $seg_count <= 1 && strlen( $clean ) > 800 );
        if ( empty( $segments ) || empty( $segments['items'] ) || $should_fallback ) {
            $seg = aeo_impr_segment_post_content( $post_id );
            if ( ! empty( $seg['items'] ) ) {
                $body['segments'] = $seg;
            }
        } else {
            $body['segments'] = $segments;
        }
        if ( ! empty( $license_key ) && 'active' === $license_status ) { $body['license_key'] = $license_key; }

        // --- DEBUG LOG START ---
        $debug_log = array(
            'timestamp' => current_time('mysql'),
            'post_id'   => $post_id,
            'request'   => array(
                'segments_count' => isset($body['segments']['items']) ? count($body['segments']['items']) : 0,
                'segments_sample' => isset($body['segments']['items']) ? array_slice($body['segments']['items'], 0, 5) : [],
                'post_content_len' => strlen($clean),
            ),
            'response'  => null,
            'mapping'   => array(),
        );
        // --- DEBUG LOG END ---

    $timeout = apply_filters( 'aeo_analysis_timeout', 60 );
        $retries = (int) apply_filters( 'aeo_analysis_retries', 0 ); // default no retries
        $attempt = 0; $last_error = ''; $response = null;
        do {
            $attempt++;
            $args = [
                'method'  => 'POST',
                'headers' => [
                    'Authorization' => 'Bearer ' . $supabase_anon_key,
                    'apikey'        => $supabase_anon_key,
                    'Content-Type'  => 'application/json',
                ],
                'body'    => wp_json_encode( $body ),
                'timeout' => $timeout,
            ];
            $response = wp_remote_post( $api_url, $args );
            if ( is_wp_error( $response ) ) {
                $last_error = $response->get_error_message();
                if ( $attempt <= ( $retries + 1 ) && preg_match( '/cURL error 28|timed out|timeout/i', $last_error ) ) {
                    do_action( 'aeo_analysis_retry_sleep', $attempt );
                        if ( ! has_action( 'aeo_analysis_retry_sleep' ) ) { usleep( 250000 * $attempt ); }
                    continue;
                }
                break;
            }
            break;
        } while ( $attempt <= ( $retries + 1 ) );
        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'connection_failed', $last_error );
        }
    $code = wp_remote_retrieve_response_code( $response );
    $body_raw = wp_remote_retrieve_body( $response );

        $data = json_decode( $body_raw, true );
        
        // --- DEBUG LOG UPDATE ---
        $debug_log['response'] = array(
            'code' => $code,
            'raw_body_len' => strlen($body_raw),
            'data_keys' => is_array($data) ? array_keys($data) : [],
        );
        // --- DEBUG LOG END ---

        if ( 200 !== $code ) {
            $msg = isset( $data['message'] ) ? $data['message'] : 'Unknown API error.';
            // Save debug log even on error
            update_post_meta( $post_id, '_aeo_debug_log', $debug_log );
            return new WP_Error( 'api_error', $msg );
        }
        $applied = aeo_apply_analysis_payload( $post_id, $data, $segments, $debug_log );
        if ( is_wp_error( $applied ) ) {
            return $applied;
        }
        return $applied;
    }
}

/**
 * Schedule auto-analysis on initial publish (draft/pending/future -> publish).
 */
function aeo_maybe_schedule_auto_analysis( $new_status, $old_status, $post ) {
    if ( 'publish' !== $new_status || 'publish' === $old_status ) { return; }
    if ( 'post' !== $post->post_type ) { return; }
    // Suspend auto-analysis during bulk batch; defer instead.
    if ( get_option( 'aeo_bulk_active', false ) ) {
        $deferred = get_option( 'aeo_bulk_deferred_auto', array() );
        $deferred[ $post->ID ] = time();
        update_option( 'aeo_bulk_deferred_auto', $deferred, false );
        return;
    }
    if ( (int) get_option( 'aeo_auto_analyze', 0 ) !== 1 ) { return; }
    if ( 'active' !== get_option( 'aeo_license_status' ) ) { return; }
    // Skip auto-analysis if manual edits were saved to avoid overwriting user content.
    if ( get_post_meta( $post->ID, '_aeo_manual_edits', true ) ) { return; }
    $lock = 'aeo_auto_analyze_lock_' . $post->ID;
    if ( get_transient( $lock ) ) { return; }
    set_transient( $lock, 1, 5 * MINUTE_IN_SECONDS );
    // Mark as queued for UI
    update_post_meta( $post->ID, '_aeo_auto_status', 'queued' );
    $delay = (int) apply_filters( 'aeo_auto_analysis_schedule_delay', 5 ); // faster initial delay
    $random_max = (int) apply_filters( 'aeo_auto_analysis_random_delay_max', 5 );
    $scheduled = wp_schedule_single_event( time() + $delay + wp_rand( 0, $random_max ), 'aeo_run_auto_analysis_event', [ $post->ID ] );
    // Fallback: if scheduling failed (rare) run inline (may block publish) – optional
    if ( ! $scheduled ) {
        update_post_meta( $post->ID, '_aeo_auto_status', 'running' );
        $res = aeo_perform_analysis( $post->ID, true );
        if ( ! is_wp_error( $res ) ) {
            update_post_meta( $post->ID, '_aeo_auto_status', 'done' );
        } else {
            update_post_meta( $post->ID, '_aeo_auto_status', 'failed' );
        }
    }
}
add_action( 'transition_post_status', 'aeo_maybe_schedule_auto_analysis', 10, 3 );

/**
 * Cron event handler for auto-analysis.
 */
function aeo_run_auto_analysis_event_cb( $post_id ) {
    if ( 'post' !== get_post_type( $post_id ) ) { return; }
    if ( 'active' !== get_option( 'aeo_license_status' ) ) { return; }
    if ( (int) get_option( 'aeo_auto_analyze', 0 ) !== 1 ) { return; }
    // Respect manual edits: do not overwrite manually saved content.
    if ( get_post_meta( $post_id, '_aeo_manual_edits', true ) ) { return; }
    update_post_meta( $post_id, '_aeo_auto_status', 'running' );
    $res = aeo_perform_analysis( $post_id, true );
    if ( ! is_wp_error( $res ) ) {
        $completed = get_option( 'aeo_auto_analysis_completed', [] );
        if ( ! is_array( $completed ) ) { $completed = []; }
        $completed[ $post_id ] = time();
        update_option( 'aeo_auto_analysis_completed', $completed, false );
        update_post_meta( $post_id, '_aeo_auto_status', 'done' );
    } else {
        // Automatic single retry logic
        $retried = get_post_meta( $post_id, '_aeo_auto_retry', true );
        if ( empty( $retried ) ) {
            update_post_meta( $post_id, '_aeo_auto_retry', 1 );
            update_post_meta( $post_id, '_aeo_auto_status', 'queued' );
            // Schedule retry after short backoff (30–40s)
            $delay = 30 + wp_rand( 0, 10 );
            wp_schedule_single_event( time() + $delay, 'aeo_run_auto_analysis_event', [ $post_id ] );
        } else {
            update_post_meta( $post_id, '_aeo_auto_status', 'failed' );
        }
    }
}
add_action( 'aeo_run_auto_analysis_event', 'aeo_run_auto_analysis_event_cb' );

/**
 * Admin notice listing newly auto-analyzed posts.
 */
function aeo_admin_notice_auto_analysis() {
    if ( ! current_user_can( 'edit_posts' ) ) { return; }
    $completed = get_option( 'aeo_auto_analysis_completed', [] );
    if ( empty( $completed ) || ! is_array( $completed ) ) { return; }
    foreach ( $completed as $pid => $ts ) {
        $p = get_post( $pid ); if ( ! $p ) { continue; }
        echo '<div class="notice notice-success is-dismissible"><p>'
            . esc_html( sprintf(
                /* translators: %s: post title */
                __( 'Automatic AEO analysis completed for "%s".', 'anslift' ),
                $p->post_title
            ) )
            . '</p></div>';
    }
    delete_option( 'aeo_auto_analysis_completed' );
}
add_action( 'admin_notices', 'aeo_admin_notice_auto_analysis' );


/**
 * Helpers: sanitize schema text and track FAQ schema emission per post.
 */
if ( ! function_exists( 'aeo_schema_clean_text' ) ) {
    function aeo_schema_clean_text( $text ) {
        if ( ! is_string( $text ) ) {
            return '';
        }
        $text = wp_strip_all_tags( $text );
        $text = preg_replace( '/\s+/', ' ', $text );
        return trim( $text );
    }
}

if ( ! function_exists( 'aeo_schema_is_faq_printed' ) ) {
    function aeo_schema_is_faq_printed( $post_id ) {
        $post_id = (int) $post_id;
        if ( $post_id < 1 ) {
            return false;
        }
        if ( empty( $GLOBALS['aeo_faq_schema_printed'] ) || ! is_array( $GLOBALS['aeo_faq_schema_printed'] ) ) {
            return false;
        }
        return ! empty( $GLOBALS['aeo_faq_schema_printed'][ $post_id ] );
    }
}

if ( ! function_exists( 'aeo_schema_mark_faq_printed' ) ) {
    function aeo_schema_mark_faq_printed( $post_id ) {
        $post_id = (int) $post_id;
        if ( $post_id < 1 ) {
            return;
        }
        if ( empty( $GLOBALS['aeo_faq_schema_printed'] ) || ! is_array( $GLOBALS['aeo_faq_schema_printed'] ) ) {
            $GLOBALS['aeo_faq_schema_printed'] = array();
        }
        $GLOBALS['aeo_faq_schema_printed'][ $post_id ] = true;
    }
}

/**
 * Inject Schema Markup into the head of single posts.
 */
function aeo_plugin_inject_schema() {
    // Only run on single post pages
    if ( ! is_single() ) {
        return;
    }

    $post_id = get_the_ID();
    if ( ! $post_id ) {
        return;
    }
    
    // Check if the post has been analyzed by our plugin
    if ( ! get_post_meta( $post_id, '_aeo_plugin_analyzed', true ) ) {
        return;
    }

    $overview_raw = get_post_meta( $post_id, '_aeo_ai_overview', true );
    $overview     = aeo_schema_clean_text( (string) $overview_raw );
    $faqs         = get_post_meta( $post_id, '_aeo_faqs', true );

    if ( '' === $overview || ! is_array( $faqs ) ) {
        return;
    }

    $faq_schema_already_printed = aeo_schema_is_faq_printed( $post_id );
    $faq_questions = array();
    if ( ! $faq_schema_already_printed ) {
        // Prepare FAQ schema questions (skip empty/invalid entries)
        foreach ( $faqs as $faq ) {
            if ( ! is_array( $faq ) ) {
                continue;
            }
            $question = aeo_schema_clean_text( isset( $faq['question'] ) ? $faq['question'] : '' );
            $answer   = aeo_schema_clean_text( isset( $faq['answer'] ) ? $faq['answer'] : '' );
            if ( '' === $question || '' === $answer ) {
                continue;
            }
            $faq_questions[] = array(
                '@type' => 'Question',
                'name' => $question,
                'acceptedAnswer' => array(
                    '@type' => 'Answer',
                    'text' => $answer,
                ),
            );
        }

        if ( empty( $faq_questions ) ) {
            return;
        }
    }
    
    // Full Schema structure
    $graph = array(
        // First, the main article schema with our description
        array(
            '@type' => 'BlogPosting', // Or 'Article'
            '@id' => get_permalink( $post_id ) . '#article',
            'headline' => aeo_schema_clean_text( get_the_title( $post_id ) ),
            'description' => $overview, // Injecting the AI Overview
            'author' => array(
                '@type' => 'Person',
                'name' => aeo_schema_clean_text( get_the_author() ),
            ),
            'datePublished' => get_the_date( 'c', $post_id ),
            'dateModified' => get_the_modified_date( 'c', $post_id ),
            'mainEntityOfPage' => array(
                '@type' => 'WebPage',
                '@id' => get_permalink( $post_id ),
            ),
        ),
    );

    // Second, the FAQPage schema (skip if already printed elsewhere for this post).
    if ( ! $faq_schema_already_printed ) {
        $graph[] = array(
            '@type' => 'FAQPage',
            'mainEntity' => $faq_questions,
        );
    }

    $schema = array(
        '@context' => 'https://schema.org',
        '@graph' => $graph,
    );

    echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) . '</script>';

    if ( ! $faq_schema_already_printed ) {
        aeo_schema_mark_faq_printed( $post_id );
    }
}
add_action('wp_head', 'aeo_plugin_inject_schema');


if ( ! defined( 'AEO_LICENSE_API_URL' ) ) {
    define( 'AEO_LICENSE_API_URL', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/activate-license' );
}
if ( ! defined( 'AEO_LICENSE_DEACTIVATE_API_URL' ) ) {
    define( 'AEO_LICENSE_DEACTIVATE_API_URL', 'https://evtgpjupcizvmiidmhif.supabase.co/functions/v1/deactivate-license' );
}

function aeo_plugin_handle_license_activation() {
    check_ajax_referer('aeo_license_nonce', 'nonce');

    if (!isset($_POST['license_key']) || empty($_POST['license_key'])) {
        wp_send_json_error(['message' => 'Please provide a license key.']);
    }

    $license_key = sanitize_text_field( wp_unslash( $_POST['license_key'] ) );
    $site_url = home_url();

    $api_url = AEO_LICENSE_API_URL;
    $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I'; 

    $response = wp_remote_post($api_url, [
        'method' => 'POST',
        'headers' => [
            'Authorization' => 'Bearer ' . $supabase_anon_key,
            'apikey'        => $supabase_anon_key,
            'Content-Type'  => 'application/json'
        ],
    'body' => wp_json_encode( [ 'license_key' => $license_key, 'site_url' => $site_url ] ),
        'timeout' => 60,
    ]);

    if (is_wp_error($response)) {
        wp_send_json_error(['message' => $response->get_error_message()]);
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $body = json_decode(wp_remote_retrieve_body($response), true);

    if ($response_code === 200) {
        update_option('aeo_license_key', $license_key);
        update_option('aeo_license_status', 'active');
        delete_option('aeo_license_error');
    // Automatically enable auto analyze upon activation if not already enabled
    add_option('aeo_auto_analyze', 1, '', false);
        wp_send_json_success(['message' => $body['message']]);
    } else {
        update_option('aeo_license_status', 'inactive');
        update_option('aeo_license_error', $body['message'] ?? 'An unknown error occurred.');
        wp_send_json_error(['message' => $body['message'] ?? 'An unknown error occurred.']);
    }
}
add_action('wp_ajax_aeo_activate_license', 'aeo_plugin_handle_license_activation');

function aeo_plugin_handle_license_deactivation() {
    check_ajax_referer('aeo_license_nonce', 'nonce');
    $license_key = get_option('aeo_license_key');
    $site_url = home_url();

    // Even if the remote call fails, we deactivate locally.
    delete_option('aeo_license_status');
    delete_option('aeo_license_error');
    // Remove stored license key so subsequent analyzes don't send it
    delete_option('aeo_license_key');

    // Optional remote deactivation endpoint (adjust if you have a specific function like "deactivate-license")
    $api_url = AEO_LICENSE_DEACTIVATE_API_URL;
    $supabase_anon_key = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImV2dGdwanVwY2l6dm1paWRtaGlmIiwicm9zZSI6ImFub24iLCJpYXQiOjE3NTUzMjE3ODIsImV4cCI6MjA3MDg5Nzc4Mn0.6BjwU-cf-lD15EXFOqmZVf1vGcFT4YWk5RYC2n-Bk9I'; // <-- IMPORTANT

    wp_remote_post($api_url, [
        'method' => 'POST',
        'headers' => [      
            'Authorization' => 'Bearer ' . $supabase_anon_key,
            'apikey'        => $supabase_anon_key,
            'Content-Type'  => 'application/json'
        ],
    'body' => wp_json_encode( [ 'license_key' => $license_key, 'site_url' => $site_url ] ),
        'timeout' => 60,
    ]);

    wp_send_json_success(['message' => 'License deactivated.']);
}
add_action('wp_ajax_aeo_deactivate_license', 'aeo_plugin_handle_license_deactivation');

/* ===================== BULK ANALYSIS IMPLEMENTATION ===================== */

// Register bulk action in post list
function aeo_register_bulk_action( $bulk_actions ) {
    $bulk_actions['aeo_bulk_analyze'] = __( 'Bulk Analyze for AEO', 'anslift' );
    return $bulk_actions;
}
add_filter( 'bulk_actions-edit-post', 'aeo_register_bulk_action' );

// Handle bulk action submission
function aeo_handle_bulk_action( $redirect_url, $action, $post_ids ) {
    if ( 'aeo_bulk_analyze' !== $action ) { return $redirect_url; }
    if ( ! current_user_can( 'edit_others_posts' ) ) { return $redirect_url; }
    if ( 'active' !== get_option( 'aeo_license_status' ) ) { return add_query_arg( 'aeo_bulk_error', 'license', $redirect_url ); }
    if ( get_option( 'aeo_bulk_active', false ) ) { return add_query_arg( 'aeo_bulk_error', 'active', $redirect_url ); }
    $result = aeo_bulk_initialize_queue( (array) $post_ids );
    if ( ! $result || empty( $result['total'] ) ) { return add_query_arg( 'aeo_bulk_error', 'empty', $redirect_url ); }
    // Schedule first batch quickly if active (fallback path when not using AJAX start)
    if ( $result['active'] && ! wp_next_scheduled( 'aeo_run_bulk_batch_event' ) ) {
        wp_schedule_single_event( time() + 1, 'aeo_run_bulk_batch_event' );
    }
    return add_query_arg( array( 'aeo_bulk_started' => 1, 'aeo_bulk_total' => $result['total'] ), $redirect_url );
}
add_filter( 'handle_bulk_actions-edit-post', 'aeo_handle_bulk_action', 10, 3 );

// Bulk batch interval constant (seconds)
if ( ! defined( 'AEO_BULK_BATCH_INTERVAL' ) ) {
    define( 'AEO_BULK_BATCH_INTERVAL', 15 );
}

// Cron event callback: process up to 10 queued posts sequentially.
function aeo_run_bulk_batch_event_cb() {
    if ( ! get_option( 'aeo_bulk_active', false ) ) { return; }
    $license_active = ( 'active' === get_option( 'aeo_license_status' ) );
    $cancel = get_option( 'aeo_bulk_cancel_requested', false );
    $queue = get_option( 'aeo_bulk_queue', array() );
    $processed = (int) get_option( 'aeo_bulk_processed', 0 );
    $failures  = (int) get_option( 'aeo_bulk_failures', 0 );
    $skipped   = (int) get_option( 'aeo_bulk_skipped', 0 );
    $total     = (int) get_option( 'aeo_bulk_total', 0 );

    if ( $cancel || ! $license_active ) {
        // Mark remaining as canceled
        foreach ( $queue as $pid ) {
            update_post_meta( $pid, '_aeo_bulk_status', 'canceled' );
            $processed++; // Count toward total completion
        }
        $queue = array();
    } else {
        $batch_size = (int) apply_filters( 'aeo_bulk_batch_size', 10 ); // Default batch size is 10
        $batch = array_splice( $queue, 0, $batch_size ); // Up to $batch_size sequentially
        foreach ( $batch as $pid ) {
            $pid = (int) $pid;
            if ( $pid < 1 ) { continue; }
            if ( get_post_type( $pid ) !== 'post' ) {
                update_post_meta( $pid, '_aeo_bulk_status', 'failed' );
                $failures++; $processed++; continue;
            }
            // Respect manual edits: skip posts manually edited to avoid overwriting user content.
            if ( get_post_meta( $pid, '_aeo_manual_edits', true ) ) {
                update_post_meta( $pid, '_aeo_bulk_status', 'skipped' );
                $skipped++; $processed++; continue;
            }
            if ( get_post_meta( $pid, '_aeo_plugin_analyzed', true ) ) {
                update_post_meta( $pid, '_aeo_bulk_status', 'skipped' );
                $skipped++; $processed++; continue;
            }
            update_post_meta( $pid, '_aeo_bulk_status', 'running' );
            $result = aeo_perform_analysis( $pid, false );
            if ( is_wp_error( $result ) ) {
                $msg = $result->get_error_message();
                $retry_key = 'aeo_bulk_retry_' . $pid;
                $already_retried = (bool) get_transient( $retry_key );
                if ( ! $already_retried && preg_match( '/timed out|timeout|cURL error 28/i', $msg ) ) {
                    // Requeue once at end
                    set_transient( $retry_key, 1, 30 * MINUTE_IN_SECONDS );
                    update_post_meta( $pid, '_aeo_bulk_status', 'queued' );
                    $queue[] = $pid; // push to end
                    continue; // Don't increment processed yet
                } else {
                    update_post_meta( $pid, '_aeo_bulk_status', 'failed' );
                    $failures++; $processed++;
                }
            } else {
                update_post_meta( $pid, '_aeo_bulk_status', 'done' );
                $processed++;
            }
        }
    }

    // Persist updated queue & counters
    update_option( 'aeo_bulk_queue', $queue, false );
    update_option( 'aeo_bulk_processed', $processed, false );
    update_option( 'aeo_bulk_failures', $failures, false );
    update_option( 'aeo_bulk_skipped', $skipped, false );

    // If there are still items queued schedule the next batch run (this was missing, causing perpetual 'Queued').
    if ( ! empty( $queue ) ) {
        $interval = (int) apply_filters( 'aeo_bulk_batch_interval', defined( 'AEO_BULK_BATCH_INTERVAL' ) ? AEO_BULK_BATCH_INTERVAL : 15 );
        // Avoid duplicate scheduling if something else already queued it (defensive).
        if ( ! wp_next_scheduled( 'aeo_run_bulk_batch_event' ) ) {
            wp_schedule_single_event( time() + max( 1, $interval ), 'aeo_run_bulk_batch_event' );
        }
    }

    if ( empty( $queue ) ) {
        // Finalize
        aeo_bulk_cleanup_options();
        // Process deferred auto-analysis posts after bulk
        $deferred = get_option( 'aeo_bulk_deferred_auto', array() );
        if ( ! empty( $deferred ) && is_array( $deferred ) ) {
            foreach ( array_keys( $deferred ) as $d_pid ) {
                $d_pid = (int) $d_pid; if ( $d_pid < 1 ) { continue; }
                if ( 'active' === get_option( 'aeo_license_status' ) && (int) get_option( 'aeo_auto_analyze', 0 ) === 1 ) {
                    update_post_meta( $d_pid, '_aeo_auto_status', 'queued' );
                    wp_schedule_single_event( time() + wp_rand( 5, 25 ), 'aeo_run_auto_analysis_event', array( $d_pid ) );
                }
            }
            delete_option( 'aeo_bulk_deferred_auto' );
        }
        set_transient( 'aeo_bulk_complete_summary', array(
            'total'    => $total,
            'processed'=> $processed,
            'failures' => $failures,
            'skipped'  => $skipped,
            'ended'    => time(),
            'canceled' => (bool) $cancel || ! $license_active,
        ), 30 * MINUTE_IN_SECONDS );
    }
}
add_action( 'aeo_run_bulk_batch_event', 'aeo_run_bulk_batch_event_cb' );

// AJAX: Poll bulk status summary & per-post statuses
function aeo_ajax_bulk_poll() {
    check_ajax_referer( 'aeo_bulk_nonce', 'nonce' );
    if ( ! current_user_can( 'edit_posts' ) ) { wp_send_json_error(); }
    // Sanitize incoming IDs array from POST without directly accessing $_POST
    $ids_input = filter_input( INPUT_POST, 'ids', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
    if ( ! is_array( $ids_input ) ) { $ids_input = array(); }
    $ids = array_map( 'absint', $ids_input );
    $map = array();
    foreach ( $ids as $pid ) {
        $map[ $pid ] = get_post_meta( $pid, '_aeo_bulk_status', true );
    }
    $total     = (int) get_option( 'aeo_bulk_total', 0 );
    $processed = (int) get_option( 'aeo_bulk_processed', 0 );
    $failures  = (int) get_option( 'aeo_bulk_failures', 0 );
    $skipped   = (int) get_option( 'aeo_bulk_skipped', 0 );
    $active    = (bool) get_option( 'aeo_bulk_active', false );
    $percent   = ( $total > 0 ) ? min( 100, round( ( $processed / $total ) * 100 ) ) : 0;
    wp_send_json_success( array(
        'active' => $active,
        'total' => $total,
        'processed' => $processed,
        'failures' => $failures,
        'skipped' => $skipped,
        'percent' => $percent,
        'statuses' => $map,
    ) );
}
add_action( 'wp_ajax_aeo_bulk_poll', 'aeo_ajax_bulk_poll' );

// AJAX: Cancel bulk
function aeo_ajax_bulk_cancel() {
    check_ajax_referer( 'aeo_bulk_nonce', 'nonce' );
    if ( ! current_user_can( 'edit_others_posts' ) ) { wp_send_json_error(); }
    if ( ! get_option( 'aeo_bulk_active', false ) ) { wp_send_json_error( array( 'message' => 'No active batch.' ) ); }
    update_option( 'aeo_bulk_cancel_requested', true, false );
    wp_send_json_success( array( 'message' => __( 'Cancellation requested. Finishing current post…', 'anslift' ) ) );
}
add_action( 'wp_ajax_aeo_bulk_cancel', 'aeo_ajax_bulk_cancel' );


/* ===== AJAX START (No page reload) ===== */

/**
 * Initialize bulk queue (shared by AJAX + fallback bulk action handler).
 * Returns array with keys: total, skipped, processed, queue (ids), active, failures.
 */
function aeo_bulk_initialize_queue( $post_ids ) {
    $post_ids = array_map( 'intval', (array) $post_ids );
    $post_ids = array_filter( $post_ids, function( $v ){ return $v > 0; } );
    if ( empty( $post_ids ) ) { return array(); }
    // Pull only valid posts
    $valid_posts = get_posts( array(
        'post_type'   => 'post',
        'post__in'    => $post_ids,
        'fields'      => 'ids',
        'post_status' => 'any',
        'numberposts' => -1,
    ) );
    $post_ids = array_values( array_intersect( $post_ids, $valid_posts ) );
    if ( empty( $post_ids ) ) { return array(); }
    $queue   = array();
    $skipped = 0;
    foreach ( $post_ids as $pid ) {
        if ( get_post_meta( $pid, '_aeo_plugin_analyzed', true ) ) {
            update_post_meta( $pid, '_aeo_bulk_status', 'skipped' );
            $skipped++;
        } else {
            update_post_meta( $pid, '_aeo_bulk_status', 'queued' );
            $queue[] = $pid;
        }
    }
    $total = count( $post_ids );
    $processed = $skipped; // Skipped are considered completed immediately
    update_option( 'aeo_bulk_queue', $queue, false );
    update_option( 'aeo_bulk_total', $total, false );
    update_option( 'aeo_bulk_processed', $processed, false );
    update_option( 'aeo_bulk_failures', 0, false );
    update_option( 'aeo_bulk_skipped', $skipped, false );
    update_option( 'aeo_bulk_cancel_requested', false, false );
    $active = ! empty( $queue );
    update_option( 'aeo_bulk_active', $active, false );
    update_option( 'aeo_bulk_started_at', time(), false );
    if ( ! $active ) {
        // All were skipped – finalize immediately
        set_transient( 'aeo_bulk_complete_summary', array(
            'total'    => $total,
            'processed'=> $processed,
            'failures' => 0,
            'skipped'  => $skipped,
            'ended'    => time(),
            'canceled' => false,
        ), 30 * MINUTE_IN_SECONDS );

        aeo_bulk_cleanup_options();
    }
    return array(
        'total'     => $total,
        'skipped'   => $skipped,
        'processed' => $processed,
        'failures'  => 0,
        'queue'     => $queue,
        'active'    => $active,
    );
}

function aeo_ajax_bulk_start() {
    check_ajax_referer( 'aeo_bulk_nonce', 'nonce' );
    if ( ! current_user_can( 'edit_others_posts' ) ) { wp_send_json_error( array( 'message' => 'Permission denied.' ) ); }
    if ( 'active' !== get_option( 'aeo_license_status' ) ) { wp_send_json_error( array( 'message' => 'License inactive.' ) ); }
    if ( get_option( 'aeo_bulk_active', false ) ) { wp_send_json_error( array( 'message' => 'Another bulk run is active.' ) ); }
    // Read IDs as an array without directly accessing $_POST
    $ids_raw = filter_input( INPUT_POST, 'post_ids', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
    if ( ! is_array( $ids_raw ) ) { $ids_raw = array(); }
    $ids = array_map( 'absint', $ids_raw );
    $result = aeo_bulk_initialize_queue( $ids );
    if ( empty( $result ) ) { wp_send_json_error( array( 'message' => 'No valid posts selected.' ) ); }
    if ( $result['active'] && ! wp_next_scheduled( 'aeo_run_bulk_batch_event' ) ) {
        wp_schedule_single_event( time() + 1, 'aeo_run_bulk_batch_event' );
    }
    // Return initial statuses for selected posts
    $statuses = array();
    foreach ( $ids as $pid ) {
        $pid = (int) $pid; if ( $pid < 1 ) { continue; }
        $statuses[ $pid ] = get_post_meta( $pid, '_aeo_bulk_status', true );
    }
    $percent = $result['total'] > 0 ? round( ( $result['processed'] / $result['total'] ) * 100 ) : 0;
    wp_send_json_success( array_merge( $result, array( 'percent' => $percent, 'statuses' => $statuses ) ) );
}
add_action( 'wp_ajax_aeo_bulk_start', 'aeo_ajax_bulk_start' );

// Admin notice: progress (active) & completion summary
function aeo_bulk_admin_notices() {
    $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
    if ( ! $screen || 'edit-post' !== $screen->id ) { return; }
    if ( get_option( 'aeo_bulk_active', false ) ) {
        // No active notice/progress bar is displayed during bulk runs per new UX.
        return;
    } else {
        $summary = get_transient( 'aeo_bulk_complete_summary' );
        if ( $summary ) {
            delete_transient( 'aeo_bulk_complete_summary' );
            $msg = $summary['canceled'] ? __( 'Bulk analysis canceled.', 'anslift' ) : __( 'Bulk analysis completed.', 'anslift' );
            $details = sprintf(
                /* translators: 1: total posts, 2: processed posts, 3: skipped posts, 4: failed posts */
                __( 'Total: %1$d, Processed: %2$d, Skipped: %3$d, Failures: %4$d', 'anslift' ),
                $summary['total'],
                $summary['processed'],
                $summary['skipped'],
                $summary['failures']
            );
            echo '<div class="notice notice-success is-dismissible"><p><strong>' . esc_html( $msg ) . '</strong> ' . esc_html( $details ) . '</p></div>';
        }
    }
}
add_action( 'admin_notices', 'aeo_bulk_admin_notices' );
